/* Copyright Barbara Liskov 1995 */

#include "int.h"
#include "runtime/obj_class.h"
#include "type.h"
#include "class_class.h"

extern class_ Int;
extern struct exception_s exc_overflow;
extern struct exception_s exc_zero_divide;
extern struct exception_s exc_illegal_char;
extern struct exception_s exc_negative_exponent;

#define BINARY_INT_COMP(name,op) \
       bool int_##name(int x, int y) \
       { return x op y; }

BINARY_INT_COMP(equal_,==)
BINARY_INT_COMP(similar_,==)
BINARY_INT_COMP(lt_,<)
BINARY_INT_COMP(gt_,>)
BINARY_INT_COMP(le_,<=)
BINARY_INT_COMP(ge_,>=)

int int_add_(int x, int y)
{
int temp;

	temp = x + y;
	if (x < 0 && y < 0 && temp >= 0) {
		exc = &exc_overflow;
		return;
		}
	if (x > 0 && y > 0 && temp <= 0) {
		exc = &exc_overflow;
		return;
		}
	return temp;
}

int int_subtract_(int x, int y)
{
int temp;

	temp = x - y;
	if (x < 0 && -y < 0 && temp >= 0) {
		exc = &exc_overflow;
		return;
		}
	if (x > 0 && -y > 0 && temp <= 0) {
		exc = &exc_overflow;
		return;
		}
	return temp;
}

#define intMin 0xffffffff80000000
#define intMax 0x7fffffff
#define intSqrtMax 46341

int int_multiply_(int x, int y)
{
int abs_x, abs_y, abs_temp, temp;

	if (x == 0 || y == 0) return 0;
	if (x == 1) return y;
	if (y == 1) return x;
	if (x == intMin || y == intMin) {
		exc = &exc_overflow;
		return;
		}
	abs_x = (x>0)?x:-x;
	abs_y = (y>0)?y:-y;
	if (abs_x < intSqrtMax && abs_y < intSqrtMax) {
		return x*y;
		}
	if (abs_x > intSqrtMax && abs_y > intSqrtMax) {
		exc = &exc_overflow;
		return;
		}
	temp = x * y;
	if (temp != intMin) {
		abs_temp = (temp>0)?temp:-temp;
		if (abs_temp > abs_x && abs_temp > abs_y) return temp;
		exc = &exc_overflow;
		return;
		}
	else {
		if (x < 0 && y > 0) return temp;
		if (x > 0 && y < 0) return temp;
		exc = &exc_overflow;
		return;
		}
}

int int_divide_(int x, int y)
{
int temp;

	if (x == intMin && y == -1) {
		exc = &exc_overflow;
		return;
		}
	if (y == 0) {
		exc = &exc_zero_divide;
		return;
		}
	if (x > 0) {
		if (y < 0) temp = (x -y -1)/y;
		else temp = x/y;
		}
	else {
		if (y > 0) temp = (x-y+1)/y;
		else temp = x/y;
		}
	return temp;
}

int int_mod_(int x, int y)
{
int temp;

	if (y == 0) {
		exc = &exc_zero_divide;
		return;
		}
	temp = x - int_divide(x,y) * y;
	return temp;
}

int int_copy_(int x)
{
    return x;
}

char int_to_char_(int x)
{
    if (x > 255 || x < 0) {
	exc = &exc_illegal_char;
	return;
	}
    return (char)x;
}

real int_to_real_(int x)
{
/*
    fevalue fv;
    fv.r = (real)x;
    return (unsigned long)fv.o;
*/
    return (float)x;
}

int int_negate_(int x)
{
   if (x == intMin) {
	exc = &exc_overflow;
	return;
	}
    return (-x);
}

static int pow_table[32] = {0,0,46340,1290,215,73,35,21,14,10,8,7,5,5,
				4,4,3,3,3,3,2,2,2,2,2,2,2,2,2,2,2,2};
int int_power_(int base, int n)
{
/* From Henry Tang */
long int tmp = 1;

  if (n<0)
    {
        exc = &exc_negative_exponent;
	return;
    }  
  if (n==0) 
    {
      return(1);
    }
  if (n==1)
    {
      return(base);
    }
/* check for overflows */
if ((n > 31) || (abs(base) > 46340) || abs(base) > pow_table[n]) {
	exc = &exc_overflow;
	return;
	}
/* do computation */
  if (n >= 1)
    { 
      while (n > 1)
        {
          if ((n%2)==0)
            {
              base *= base;
              n /= 2;
            }
          else
            {
              tmp *= base;
              n--;
            }
        }
      base *= tmp;
      return (base);
    }
}

int_abs_(int x)
{
	if (x == intMin) {
		exc = &exc_overflow;
		return;
		}
	return (x>0)?x:-x;
	}

int_to_by_(int lb, int ub, int step)
{
	fail("Not implemented yet.");
	}

int_min_(int x, int y)
{
	return x<y?x:y;
	}

int_max_(int x, int y)
{
	return x>y?x:y;
	}

void int_to_(int x, struct closure cl, int y)
{
  for (; x <= y; x++)
    cl.f(cl.env, x);
}

struct intdv_s {
    struct dv_s super;
    int (*add)(int, int);
    int (*subtract)(int, int);
    int (*multiply)(int, int);
    int (*divide)(int, int);
    int (*mod)(int, int);
    bool (*equal)(int, int);
    bool (*similar)(int, int);
    int (*copy)(int);
    bool (*lt)(int, int);
    bool (*gt)(int, int);
    bool (*le)(int, int);
    bool (*ge)(int, int);
    char (*to_char)(int);
    real (*to_real)(int);		/* possibly should be uns long */
    string (*unparse)(int);
    int (*negate)(int);
    int (*power)(int, int);
    int (*abs)(int);
    void (*to)(int, struct closure, int);
    int (*to_by)(int, int, int);
    int (*max) (int, int);
    int (*min) (int, int);
} int_methods = {
    {0, 0, STD_FOFFSET, 0, 0, normal_get_address, normal_get_class},
    int_add_,
    int_subtract_,
    int_multiply_,
    int_divide_,
    int_mod_,
    int_equal_,
    int_similar_,
    int_copy_,
    int_lt_,
    int_gt_,
    int_le_,
    int_ge_,
    int_to_char_,
    int_to_real_,
    int_unparse,
    int_negate_,
    int_power_,
    int_abs_,
    int_to_,
    int_to_by_,
    int_max_,
    int_min_
  };
      
DV Int_dh[] = { (DV)&int_methods };

void initInt()
{
    Int->dh = Int_dh;
    Int->dhsize = 1;
    int_methods.super.c = Int;
}
