/* Copyright Barbara Liskov 1995 */

#ifndef _OBJTYPE_CLASS_H
#define _OBJTYPE_CLASS_H

#include "runtime/obj_class.h"
#include "types/type.h"
#include "types/type_meth.h"
#include "types/vec_def.h"
#include "types/objtype.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct objtypedv_s {
    struct typedv_s super;
    void (*get_method)(objtype, fevalue *, string);
} *objtypedv;

typedef vec dh_template; /* dh_template = vec[dv_template] */
#define dv_fetch(DHT, I) (UNPV(vec_fetch(DHT, I), dv_template))
/*
   A "dh_template" describes the layout of a dispatch header for an objtype
   or class. The dispatch header contains a sequence of pointers to
   dispatch vectors, which each of the entries in the "dh_template"
   describes in order. For a class, the last entry describes the class
   dispatch vector.

   All "dh_template" objects are constant after initialization.

   The "dv_fetch" operator fetches the "I"th dispatch vector template
   from the header template.
*/

typedef vec dv_template; /* dv_template = vec[any] */
/*
    A "dv_template" describes the layout of a particular dispatch vector
    for the type. The type of the entry in the template vector defines
    what kind of data is found in the dispatch vector at the corresponding
    location:

    A method object: a pointer to that method.
    A parameter type: the corresponding actual type for that parameter.
    An object type or class: some object of that type or class.
    nil: an empty entry

    All "dv_template" objects are constant after initialization, so may
    be reused by subtypes.
*/

struct objtype_s {
    union {
	struct core_s inh;
	objtypedv methods;
    } hdr;

    string name;
    vec supertypes_;
    vec methods_;
    int kind;
    int hdr_size;
    dh_template dht;
    DV *surr_DH;
    DV *full_surr_DH;
    DV *unswizzled_DH;
    string overview;
};

/*
   The field "methods_" only describes those methods that are new with
   this type. Methods inherited (by interface inheritance) from supertypes
   are not listed in this field, unless their names or signatures have
   been changed. Not that method renaming is not supported at present.

   The field "kind" allows a quick determination of what kind of type
   this is.

   A "hdr_size" of -1 indicates that the header size has not yet been
   computed. If not, "surr_DH" and its cronies will be 0.
*/

#define OBJTYPE_SLOTS 9
#define OBJTYPE_BITFIELD 0x117
/*
   These two definitions must be kept in sync with the definition of
   "struct objtype_s". Because of circularities in the bootstrap
   process, they cannot be precomputed.
*/


extern dh_template get_dh_template(objtype t);
/*
  Returns the dh_template for object type t.  Computes it, if not
  already computed.
*/

/* See "method.h" for the specification of the type "method. */
struct method_s {
    struct core_s hdr;
    string name;
    vec arguments; /* vec[formal] */
    vec signals;   /* vec[signal] */
    vec returns;   /* vec[type] */
    vec extra_args;/* vec[type] */
    bool iter;
    bool parameterized;
    int index;
    string spec;
    type self_type;
};

#define METHOD_SLOTS 8
#define METHOD_BITFIELD 0xDF

method new_method();
/* Returns a "method" object in which only the "hdr" field has been
   initialized. */

typedef struct formal_s {
    struct core_s hdr;
    string name;
    type t;
} *formal;

typedef struct signal_s {
    struct core_s hdr;
    string name;
    vec returns;
} *signal_;

extern bool isa_objtype(obj o);
/* Return true iff "o" is a type object.  Remember, all classes
   are types and therefore this routine will return true
   for class objects as well.
*/

extern bool isa_class(obj o);
/* Return true iff "o" is a class object.
*/

extern objtype new_objtype();
extern signal_ new_signal();
extern formal new_formal();

extern class_ Formal, Signal;

extern struct dv_s formal_methods;

void init_type(objtype t);
/* Initialize the header of "t", which must be a completely
   uninitialized object of class "ObjType".
   This procedure is provided to handle circularities in
   the bootstrap process. See also "init_class".
*/

extern void init_ObjType();
/*
   Initialize "ObjType" properly after creation.
*/

extern method getClass_method;
/*
  A "method" describing the "getClass" method. Used for type
  initialization, and shared by all type descriptors.
*/

/* The following methods actually require a first argument that is
   an objtype. They should only be used to populate the dispatch
   tables of subclasses. */
extern bool objtype_equal_(type t1, type t2);
extern void objtype_methods_(type t, struct closure);
extern bool objtype_isSubtype_(type t1, type t2);
extern string objtype_name_(type);
extern string objtype_unparse_(type);
extern void objtype_supertypes_(type, struct closure);
extern bool objtype_isPrimitive_(type);
extern void objtype_get_method_(objtype, fevalue *__retvals, string name);
extern int objtype_kind_(type);

#ifndef __cplusplus
extern objtype type_as_objtype(type);
#else

/* Must match non-inlined version in objtype.c */
static inline objtype type_as_objtype(type t)
{
#ifndef NDEBUG
    class_ tc = get_obj_class(type_as_obj(t));
    assert(tc == ObjType || tc == Instn || tc == Class ||
    tc == ClassInstn || tc == PClass || tc == PType);
#endif
    return (objtype)(BUMP(objtype, t, (DV)t->methods));
}
#endif

#ifdef __cplusplus
}
#endif

#endif /* _OBJTYPE_CLASS_H */
