/* Copyright Barbara Liskov 1995 */

#ifndef _PTYPE_CLASS_H
#define _PTYPE_CLASS_H

/*
    This include contains private information for both the "ptype"
    and "instn" implementations, which are heavily intertwined.
*/

#include "types/list.h"
#include "types/ptype.h"
#include "objtype_class.h"

/*

Revelation: instantiations are not required to be valid. The compiler
makes use of this fact because method declarations are not available
during the pass that creates the instantiations. Therefore, the
parameters to an instantiation cannot be checked at create time.

The procedures here provide a back door for instantiations to be
[emacs suspend]created in two phases: first created, then later validated. Invoking
an operation on an invalid instantiation causes unpredictable results.

*/

extern instn new_instn(ptype pt, vec args);
extern void init_instn(instn ins, ptype pt, vec args);

/*
    Act like "instantiate", but generate an "instn" that is not
    yet valid. Signal "bounds" if the length of the argument list
    does not match the parameter list.
*/

extern void complete_instn(objtype, ptype, vec);
/*

    Complete an instantiation. Signal "not_possible(type, string)" if
    the arguments do not match their parameters, just as
    "ptype_instantiate" does.

*/

extern bool validate_instn(objtype, ptype, vec);
/*

    Validate an instantiation. Signal "not_possible(type, string)" if
    the arguments do not match their parameters, just as
    "ptype_instantiate" does.

*/

extern method method_instantiate(method m, type self_type, vec params, 
						vec pargs);
/*
     Instantiate a method.
*/

typedef struct ptypedv_s {
    struct objtypedv_s super;
    instn (*instantiate)(ptype, vec types);
    void (*parameters)(ptype, struct closure);
} *ptypedv;


struct ptype_s {
    union {
	struct objtype_s inh;
	ptypedv methods;
    } hdr;
    vec params; /* vec[Param] */
    list instances;
};

#define PTYPE_SLOTS (2 + OBJTYPE_SLOTS)
#define PTYPE_BITFIELD ((0x3 << OBJTYPE_SLOTS) | OBJTYPE_BITFIELD)

extern void init_ptype(ptype pt);
/* Set up the headers of a ptype object. "params" is a "vec[Param]"
   and may not be modified after passing it into this routine.
*/

typedef struct instndv_s {
    struct objtypedv_s super;
    ptype (*get_ptype)(instn);
    void (*pargs)(instn, struct closure);
} *instndv;

struct instn_s {
    union {
	struct objtype_s inh;
	instndv methods;
    } hdr;
    ptype ptype_;
    vec pargs;      /* vec[Type] */
    bool valid;
};

#define INSTN_SLOTS (OBJTYPE_SLOTS + 3)
#define INSTN_BITFIELD ((0x3 << OBJTYPE_SLOTS) | OBJTYPE_BITFIELD)

extern void init_instn_name(ptype pt, objtype t, vec pargs);
/*
   Set the name of "t" as if it were an instantiation of
   the ptype "pt", with parameters "pargs".
*/

extern void ptype_parameters_(ptype, struct closure);
extern instn ptype_instantiate_(ptype pt, vec args);
extern type instantiate(type t, type self_type, vec fparms, vec aparms);


#endif /* _PTYPE_CLASS_H */
