/* Copyright Barbara Liskov 1995 */

#include "string_class.h"
#include "objtype_class.h"
#include "runtime/alloc.h"
#include "common/other_unix.h"
#include <stdlib.h>

extern struct exception_s exc_negative_size;

static int string_length_(string s);
static string string_concat_(string s1, string s2);
static int th_strncmp(unsigned char *x, int xlen, unsigned char *y, int ylen);

struct stringdv_s string_methods = {
    { 0, 0, STD_FOFFSET, 0, 0,
                     normal_get_address,
                     normal_get_class},
                     string_length,
                     string_concat,
                     string_equal,
                     string_chars,
                     string_fetch_,
                     string_unparse,
                     string_empty_,
                     string_first,
                     string_rest,
                     string_append,
                     string_extract,
                     string_lt,
                     string_gt,
                     string_le,
                     string_ge,
                     string_similar,
                     string_copy
                   };

string string_new(char const s[])
{
    int len = strlen(s);
    return string_newn(s, len);
}

string string_newn(char const s[], int len)
{
    string ret;
    if (normal_heap)
      ret = NEWSTRING(len);
    else
      ret = NEW_META_STRING(len);
    init_obj_hdr_prim(&ret->hdr.inh, (len + sizeof(int) + 1 +
				  (sizeof(fevalue) - 1))/sizeof(fevalue),
		      OBJ_BF_ALLDATA,
		      (DV)&string_methods);
    if (len) memcpy(ret->chars, s, len);
    ret->chars[len] = 0;
    ret->size = len;
    return ret;
}

int string_length(string s)
{
    FIX_FAST(s,string,string_methods);
    return string_length_(s);
} 

static int string_length_(string s)
{
    return s->size;
} 

string string_concat(string s1, string s2)
{
    FIX_FAST(s1,string,string_methods);
    FIX_FAST(s2,string,string_methods);
    return string_concat_(s1,s2);
}

static string string_concat_(string s1, string s2)
{
    int l1 = s1->size, l2 = s2->size;
    int len = l1 + l2;
    string ret;
    if (normal_heap)
      ret = NEWSTRING(len);
    else
      ret = NEW_META_STRING(len);
    init_obj_hdr_prim(&ret->hdr.inh,
		      (sizeof(int) + len + 1 + sizeof(fevalue) - 1)/
		      sizeof(fevalue),
		      OBJ_BF_ALLDATA,
		      (DV)&string_methods);
    if (len) {
      bcopy(s1->chars, ret->chars, l1);
      bcopy(s2->chars, ret->chars + l1, l2);
    }
    ret->chars[len] = 0;
    ret->size = len;
    return ret;
}

string string_ctos(char c)
{
    char buf[2];
    buf[0] = c;
    buf[1] = 0;
    return string_newn(buf, 1);
}

string string_empty()
{
    return string_newn(0, 0);
}

bool string_empty_(string s)
{
    if (s->size == 0) return TRUE;
    else return FALSE;
}

char string_fetch(string s, int i)
{
    FIX_FAST(s,string,string_methods);
    if (i>=0 && i<s->size) return s->chars[i];
    exc = &exc_bounds;
}

char string_fetch_(string s, int i)
{
    FIX_FAST(s,string,string_methods);
    if (i>=1 && i<=s->size) return s->chars[i-1];
    exc = &exc_bounds;
}

bool string_equal(string s1, string s2)
{
    FIX_FAST(s1, string, string_methods);
    FIX_FAST(s2, string, string_methods);
    if (s1->size != s2->size) return FALSE;
    return !(bcmp(s1->chars, s2->chars, s1->size));
}

void string_chars(string s, struct closure cl)
{
    FIX_FAST(s, string, string_methods);
    {
	int i = s->size;
	char *c = s->chars;
	obj env = cl.env;
	void (*f)(obj, char) = (void (*)(obj, char))cl.f;
	while (i--) {
	    f(env, *c++);
	    CHECK_BREAK_EXC;
	}
    }
}

/*
string string_unparse(string s)
{
	FIX_FAST(s,string,string_methods);
	return s;
}
*/

string string_first(string s, int n)
{
	FIX_FAST(s,string,string_methods);
	if (n < 1 || n > s->size) {
		exc = &exc_bounds;
		return;
		}
	return string_newn(s->chars, n);
}

string string_rest(string s, int n)
{
	FIX_FAST(s,string,string_methods);
	if (n < 1 || n > s->size) {
		exc = &exc_bounds;
		return;
		}
	return string_newn(&s->chars[n-1], (s->size - (n-1)));
	
}

string string_append(string s, char c)
{
string ret;
	FIX_FAST(s,string,string_methods);
	ret = string_newn(s->chars, s->size+1);
	ret->chars[s->size] = c;
	return ret;
}

string string_extract(string s, int at, int count)
{
string ret;
int real_count;

	FIX_FAST(s,string,string_methods);
	if (count < 0) {
		exc = &exc_negative_size;
		return;
		}
	if (at < 1 || at > s->size ){
		exc = &exc_bounds;
		return;
		}
	real_count = count;
	if ((at + count - 1) > s->size) real_count = s->size - at + 1;
	ret = string_newn(&s->chars[at-1], real_count);
}

bool string_lt(string x, string y)
{
int res;
	FIX_FAST(x,string,string_methods);
	FIX_FAST(y,string,string_methods);
	res = th_strncmp((unsigned char *)x->chars, x->size, 
				(unsigned char *)y->chars, y->size);
	if (res < 0) return TRUE;
	else return FALSE;
	}

bool string_gt(string x, string y)
{
int res;
	FIX_FAST(x,string,string_methods);
	FIX_FAST(y,string,string_methods);
	res = th_strncmp((unsigned char *)x->chars, x->size, 
				(unsigned char *)y->chars, y->size);
	if (res > 0) return TRUE;
	else return FALSE;
	}

bool string_le(string x, string y)
{
int res;

	FIX_FAST(x,string,string_methods);
	FIX_FAST(y,string,string_methods);
	res = th_strncmp((unsigned char *)x->chars, x->size, 
				(unsigned char *)y->chars, y->size);
	if (res <= 0) return TRUE;
	else return FALSE;
	}

bool string_ge(string x, string y)
{
int res;
	FIX_FAST(x,string,string_methods);
	FIX_FAST(y,string,string_methods);
	res = th_strncmp((unsigned char *)x->chars, x->size, 
				(unsigned char *)y->chars, y->size);
	if (res >= 0) return TRUE;
	else return FALSE;
	}

/*
	th_strncmp compares two c-strings, that may have embedded nulls.
	It returns 0 if the strings are equal lexicographically.
	It returns 1 if the first string is greater lexicographically
	than the second.
	It returns -1 if the second string is greater than the first.
*/

static int th_strncmp(unsigned char *x, int xlen, unsigned char *y, int ylen)
{
int res = 0;
int i;
int len = xlen<ylen ? xlen : ylen;

	for (i = 0; i < len; i++) {
		res = x[i] == y[i];
		if (res == 0) {
		  if (x[i] > y[i]) return(1);
		  return (-1);
		}
	      }
	if (xlen > ylen) return (1);
	if (ylen > xlen) return (-1);
	return (0);
	}

bool string_similar(string s1, string s2)
{
    FIX_FAST(s1, string, string_methods);
    FIX_FAST(s2, string, string_methods);
    if (s1->size != s2->size) return FALSE;
    return !(bcmp(s1->chars, s2->chars, s1->size));
}

string string_copy(string s)
{
   FIX_FAST(s, string, string_methods);
   return string_newn(s->chars, s->size);
}


char const *string_charp(string s)
{
    FIX_FAST(s, string, string_methods);
    return s->chars;
}

void delete_string(string s)
{
    THOR_FREE(s);
}

#include "class_class.h"

DV string_DH[] = { (DV)&string_methods };
void initString()
{
    String->dh = string_DH;
    String->dhsize = 1;
    string_methods.super.c = String;
}
