/* Copyright Barbara Liskov 1995 */

#ifndef _STRING_H
#define _STRING_H

#include "runtime/obj.h"
#include "runtime/except.h"
#include "common/basic.h"
#include "common/iter.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Overview: A "string" is an immutable sequence of characters.
   "string" is a class, so it carries with it an implementation
   as well as an interface.
*/

extern struct exception_s exc_bounds;

/* Methods */

string string_concat(string s1, string s2);

void string_chars(string s1, struct closure cl);
/* Yield the characters of "s1" to "cl", in order.
*/

int string_length(string s);
/* Return the number of characters in "s".
*/

char string_fetch(string s, int index);
/* Given that index is in [0..length(s)), return the "index"th character.
   Signal "bounds" if the index is out of bounds.
*/

char string_fetch_(string s, int index);
/* Given that index is in [1..length(s)), return the "index"th character.
   Signal "bounds" if the index is out of bounds.
*/

bool string_equal(string s1, string s2);
/* Return whether "s1 == s2" */

bool string_empty_(string s);
/* Return TRUE is the string is empty else FALSE */

bool string_lt(string x, string y);
/* Return TRUE is x is lexicographically less than y */
/* Otherwise return FALSE. */

bool string_gt(string x, string y);
/* Return TRUE is x is lexicographically greater than y */
/* Otherwise return FALSE. */

bool string_le(string x, string y);
/* Return TRUE is x is lexicographically less than or equal to y */
/* Otherwise return FALSE. */

bool string_ge(string x, string y);
/* Return TRUE is x is lexicographically greater than or equal to y */
/* Otherwise return FALSE. */

bool string_similar(string s1, string s2);
/* Return TRUE is x is the same as y */


/* Creators */

string string_empty();
/* Return the empty string. */

string string_ctos(char c);
/* Return a string containing just the character "c".
*/

string string_new(char const s[]);
/* Given that "s" is a null-terminated character array, create a string
   containing the characters in "s" up to the first null character.
*/

string string_newn(char const s[], int len);
/* Given that "s" is a character array of length at least "len",
   Create a string containing the first "len" characters in "s".
*/

string string_copy(string s);
/* Return a new string with the same characters as s. */

string string_first(string s, int n);
/* Requires n to be a valid index of s else signal bounds. */
/* Return new string containing the first n characters of s. */

string string_rest(string s, int n);
/* Requires n to be a valid index of s else signal bounds. */
/* Return new string containing all the characters in s starting at index n. */

string string_append(string s, char c);
/* Return a new string with the same characters as s, with c on the end. */

string string_extract(string s, int at, int count);
/* Requires at to be a valid index of s else signal bounds. */
/* Requires count to be non-negative else signal negative_size. */
/* Return new string containing at most count characters in s starting at index at. */

void delete_string(string s);
/* Free all storage associated with "s". The string "s" may not be used
 * after this call.
 */

string string_unparse(string s);
/*
 * Unparse "s", yielding a string that is a Theta representation for
 * a string constant equal to "s". For example, the string "hello"
 * would be unparsed as "hello".
 */

string string_unparse_fancy(string s);
/*
 * Unparse "s", yielding a string that is a Theta representation for
 * a string constant equal to "s". For example, the string "hello"
 * would be unparsed as "\"hello\"".
 */

char const *string_charp(string s);
/* Return a pointer to the characters of "s". This operation
   is a back door for efficiency, and is not exposed in the
   Theta string interface. Except in cases of dire efficiency,
   it should be used instead of directly accessing the "chars" field.
*/

#define string_as_obj(s) ((obj)s)

extern void initString();
/* initialize DH of String */

#ifdef __cplusplus
}
#endif

#endif /* _STRING_H */
