/* Copyright Barbara Liskov 1995 */

#include "wr_meth.h"
#include "textwr.h"
#include "type.h"
#include "string_class.h"
#include "runtime/alloc.h"
#include "class_class.h"

#include <string.h>

struct textwr_s {
    union {
	struct textwrdv_s *methods;
	struct core_s hdr;
    } m;
    string buf;
    int end;
    int index;
};

/* This implementation actually mutates the characters in the string "buf".
   No other runtime component currently does this, and we may wish to
   outlaw such mutation for performance reasons.
   */

/* Invariant: 0 <= index <= end <= size
   Characters indexed by [0..end) are characters that
   have been written into the buffer. "index" is the current
   position of the writer.
*/

struct textwrdv_s {
    struct wrdv_s super;
    string (*toString)(textwr);
};
static struct textwrdv_s textwr_methods;

static void check_grow(textwr tw, int newend)
{
    int old_size = string_length(tw->buf);
    if (newend > tw->end) {
	if (newend > old_size) {
	    string newbuf;
	    int size = old_size * 2;
	    newbuf = NEWSTRING(size);
	    assert(String);
	    init_obj_hdr(cao(&newbuf->hdr.inh), String);
	    newbuf->size = (size + (sizeof(int) + 1 +
					(sizeof(fevalue) - 1)))/
					sizeof(fevalue);
	    newbuf->size = size;
	    memcpy((char *)string_charp(newbuf),
		   string_charp(tw->buf),
		   tw->end);
	    tw->buf = newbuf;
	}
	tw->end = newend;
    }
}

static void textwr_putChar(wr tw_, char c)
{
    textwr tw = (textwr)tw_;
    FIX_FAST(tw, textwr, textwr_methods);
    check_grow(tw, tw->index + 1);
    ((char *)string_charp(tw->buf))[tw->index] = c;
    tw->index++;
}

static void textwr_putString(wr tw_, string s)
{
    textwr tw = (textwr)tw_;
    int l = string_length(s);
    FIX_FAST(tw, textwr, textwr_methods);
    check_grow(tw, tw->index + l);
    memcpy((char *)string_charp(tw->buf) + tw->index, string_charp(s), l);
    tw->index += l;
}

static int textwr_length(wr w)
{
    textwr tw = (textwr)w;
    FIX_FAST(tw, textwr, textwr_methods);
    return tw->end;
}

string textwr_toString(textwr tw)
{
    FIX_FAST(tw, textwr, textwr_methods);
    return string_newn(string_charp(tw->buf), tw->end);
}

void textwr_seek(wr tw_, int posn)
{
    textwr tw = (textwr)tw_;
    FIX_FAST(tw, textwr, textwr_methods);
    if (posn <= tw->end && posn >= 0) 
	tw->index = posn;
}

int textwr_index(wr tw_)
{
    textwr tw = (textwr)tw_;
    FIX_FAST(tw, textwr, textwr_methods);
    return tw->index;
}

void textwr_close(wr tw_) {}

void textwr_flush(wr tw_) {}

static struct textwrdv_s textwr_methods = {
    {
	{
	    0, 0, STD_FOFFSET, 0, 0,
	    normal_get_address,
	    normal_get_class
	  },
	      textwr_putChar,
	      textwr_putString,
	      textwr_seek,
	      textwr_length,
	      textwr_index,
	      textwr_flush,
	      textwr_close
	    },
		textwr_toString
	      };

DV textwr_DH[] = {
    (DV)&textwr_methods };
     
char dummy_buf[256];
     
textwr textwr_new()
{
    textwr tw = NEW(struct textwr_s);
    init_obj_hdr_prim(&tw->m.hdr, 2, 0x1, (DV)&textwr_methods);
    tw->buf = string_newn(dummy_buf, sizeof(dummy_buf));
    /* large enough to handle many small unparsing tasks */
    tw->index = tw->end = 0;
    return tw;
}

wr textwr_as_wr(textwr tw)
{
    return (wr)tw;
}

void initTextWr()
{
    TextWr->dhsize = 1;
    TextWr->dh = textwr_DH;
    textwr_methods.super.super.c = TextWr;
}
