// Copyright 1995 Barbara Liskov

/*
\section{Two-phase commit protocol}

This file is the part of the FE manager that deals with the two-phase commit.

To do:
\begin{itemize}
\item Try to retransmit if sending a message to an OR fails.
\end{itemize}
 */

#include "common/fail.h"
#include "common/mos.h"
#include "common/Timer.h"
#include "common/ros.h"
#include "common/nos.h"
#include "common/or_set.h"
#include "common/transaction.h"
#include "common/fe_or_msg.h"
#include "common/network.h"
#include "common/xrefs.h"
#include "common/uids.h"
#include "common/unparser.h"

#include "or_or_msg.h"
#include "fe_manager.h"
#include "or.h"
#include "or_config.h"
#include "or_manager.h"
#include "tm.h"
#include "tstatus.h"
#include "mm/log.h"

#include <sys/uio.h>

extern float total_time, recv_time, send_time, validate_time;
extern Timer total_timer, recv_timer, send_timer, validate_timer;
extern Timer cumm_total_timer, cumm_recv_timer, cumm_send_timer, cumm_validate_timer;

void FE_manager::commit_trans (bool coordinator) {
    // Get rest of the info from the FE.
    // Maybe these dynamic allocations should be avoided.

    unparser unp(0);
    Transaction tx;
    tx.ros = new Ros(0); // Do not know the size of incoming ROS
    // Maybe we should use some history information?
    tx.mos = new Mos;
    tx.nos = new Nos;
    tx.participants = new OR_set;
    
    cumm_recv_timer.start();
    recv_timer.reset(); recv_timer.start();
    if (!tx.decode(net)) {
	// Give a warning or something
	delete tx.ros;
	delete tx.mos;
	delete tx.nos;
	delete tx.participants;
	return;
    }

    if (or->config->debug_level > 0) {
	id().unparse(&unp);
	if (or->config->debug_level > 1)
	    tx.unparse(&unp);
	fprintf(stderr, "Commit:\n");
	fprintf(stderr, "  Read:  %8d objs\n", tx.ros->size());
	fprintf(stderr, "  Write: %8d objs, %8d bytes\n",
		tx.mos->count(),
		tx.mos->rep_size());
	fprintf(stderr, "  New:   %8d objs, %8d bytes\n",
		tx.nos->count(),
		tx.nos->rep_size());
    }

    recv_timer.stop(); recv_time = recv_timer.elapsed();
    cumm_recv_timer.stop();

    // Check for read-only everywhere transaction
    if (tx.coordinator == 0) 
	commit_read_only(&tx);
    else // Check for single-OR transaction
	if (tx.participants->size() == 1)
	    commit_single(&tx);
	else commit_multiple(&tx, coordinator);

    // Mos and nos are being deleted by the rinfo for this transaction
    // when the objects are no longer needed in the log.
    delete tx.ros;
    delete tx.participants;
}


void FE_manager::commit_read_only(Transaction *tx) {
    or_message msg;
    Xrefs xrefs;
    Uids uids;

    ubits32 result = or->tm->commit_single(this, tx, &xrefs, &uids);
    cumm_send_timer.start();
    send_timer.start();
    // Send result to FE
    if (result == OR_COMMITTED) {
	msg.msgtype = OR_READ_OK;
	if (or->config->debug_level > 1)
	    fprintf(stderr, "Read-only transaction committed\n");
    } else {
	msg.msgtype = result;
	if (or->config->debug_level > 1)
	    fprintf(stderr, "Read-only transaction aborted\n");
    }
    msg.encode(net);
    net->flush();
    send_timer.stop(); send_time = send_timer.elapsed();
    cumm_send_timer.stop();
}

void FE_manager::commit_multiple(Transaction *tx, bool coordinator) {
    int res;
    int newcount = tx->nos->count();

    if (coordinator) {
	// Must be heap allocated since stored in coordinator's vote
	Xrefs *xrefs = new Xrefs(newcount);
	Uids  *uids  = new Uids(newcount);
	result = 0;

	res = or->tm->prepare_coord(this, tx, xrefs, uids);

	// If we already know outcome of transaction, 
	// send message to FE, and we're done.
	if (res != 0) {
	    finish_transaction(tx->tid, res);
	    return;
	}

	// Wait for transaction to complete
	mutex_commit->grab(); {
	    while (result == 0) 
		cond_commit->wait();
	    res = result;
	} mutex_commit->release();

	if (or->config->debug_level > 1)
	    printf("FE thread got transaction result; sending it to FE\n");
	finish_transaction(tx->tid, res);
    }
    else { // We are a participant
	Xrefs xrefs(newcount);
	Uids  uids(newcount);

	or_or_message or_msg;
	bool forced = TRUE;
	res = or->tm->prepare_part(this, tx, &xrefs, &uids, forced);

	// Send vote to coordinator
	or_msg.tid = tx->tid;
	if (res == OR_COMMITTED) {
	    // Send read only or read/write vote to coordinator
	    if (tx->mos->count() == 0) {
		or_msg.msgtype = OR_VOTE_READ;
	    } 
	    else {
		// Send OK vote and new objects
		or_msg.msgtype = OR_VOTE_OK;
		or_msg.u.vote_ok.count = newcount;
		or_msg.u.vote_ok.xrefs = &xrefs;
		or_msg.u.vote_ok.uids  = &uids;

		// If prepare record not forced, send to coordinator
		if (!forced) {
		    Log_Index index  = or->tstatus->get_log_index(tx->tid);
		    
		    Prepared_Log_Record *pr = 
			(Prepared_Log_Record *) or->log->fetch(index);
		    
		    if (pr != 0) {
			if (or->config->debug_level > 1)
			    printf("Sending prepare record to coordinator\n");
			or_msg.msgtype = OR_VOTE_OK_CL;
			or_msg.u.vote_ok.pr = pr;
			or_msg.u.vote_ok.index = index;
		    }
		}
	    }
	    
	    // XXX Assume that sending message succeeds
	    or->or_managers->send_message(tx->coordinator, &or_msg);
	    if (or->config->debug_level > 1)
		printf("Sent ok vote to coordinator\n");
	}
	else {
	    // Send abort vote
	    or_msg.msgtype = OR_VOTE_ABORT;
	    
	    // XXX Assume that sending message succeeds
	    or->or_managers->send_message(tx->coordinator, &or_msg);

	    if (or->config->debug_level > 1)
		printf("Sent abort vote to coordinator\n");

	    or->tm->truncate_vqueue_if_needed();
	}
    }
}

void FE_manager::commit_single(Transaction *tx) {
    int result;
    int newcount = tx->nos->count();
    Xrefs xrefs(newcount);
    Uids  uids(newcount);
    result = or->tm->commit_single(this, tx, &xrefs, &uids);
    if (result == OR_COMMITTED) {
	th_assert(xrefs.size() == newcount, "wrong number of new xrefs");
	th_assert(uids.size()  == newcount, "wrong number of new uids");
	send_commit(&xrefs, &uids);
	if (or->config->debug_level > 0)
	    fprintf(stderr, "Transaction %d Committed\n", tno);
    }
    else {
	if (or->config->debug_level > 0)
	    fprintf(stderr, "Transaction Aborted\n", tno);
	send_abort(result);
    }
    tno++;
}

void FE_manager::finish_transaction(Tid const& tid, int result) {
    bool done;

    switch (result) {
    case OR_STALEABORT:
	send_abort(result);
	if (or->config->debug_level > 0)
	    fprintf(stderr, "Transaction %d Aborted\n", tno);
	break;

    case OR_COMMITTED:
	// Send commit to FE
	Xrefs *new_xrefs;
	Uids  *new_uids;
	done = or->tstatus->get_fe_objs(tid, new_xrefs, new_uids);
	send_commit(new_xrefs, new_uids);
	// Remove from tstatus if all acks have arrived
	if (done)
	    or->tstatus->remove(tid);

	if (or->config->debug_level > 0)
	    fprintf(stderr, "Transaction %d Committed\n", tno);
	break;
    }
    tno++;
}

void FE_manager::send_commit(Xrefs *xrefs, Uids *uids) {
    or_message msg;
    int newcount = xrefs->size();

    msg.msgtype = OR_COMMITTED;
    msg.u.commit.count = newcount;
    
    // Encode the xrefs and uids
    struct iovec iov[2];
    iov[0].iov_base = (caddr_t) xrefs->as_pointer();
    iov[0].iov_len  = sizeof(Xref) * newcount;
    iov[1].iov_base = (caddr_t) uids->as_pointer();
    iov[1].iov_len  = sizeof(OR_uid) * newcount;
    
    msg.encode(net);
    net->send_vector(iov, 2);
    net->flush();
}

void FE_manager::send_abort(int msgtype) {
    or_message msg;
    msg.msgtype = msgtype;
    msg.encode(net);
    net->flush();
}
