// Copyright 1995 Barbara Liskov

/*
\section{Set of FE_info}

Contains all of the per-FE information.
*/


#ifndef _FE_INFO_SET_H
#define _FE_INFO_SET_H

#include "common/basic.h"
#include "common/array.h"
#include "fe_table.h"
#include "inv_set.h"
#include "rwlock.h"

class fe_num;
class FE_manager;

// The FE_info struct holds all per-FE information, including FE table, etc.

struct FE_info {
    fe_num       id;           // FE's id #
    FE_manager  *manager;      // FE manager thread for this FE
    FE_table    *fe_table;     // Set of objects used at FE
    Invalid_set *invalid_objs; // Set of objects that need to be invalidated 
    Collection_num  max_message;  // Highest # invalidation msg sent to FE
    Collection_num  current_message;  // Highest # invalidation msg that
				      // has been generated for this FE
    RW_lock     *lock;         // Lock on FE_info structure
};


declareArray(FE_info_array, FE_info *)

class FE_info_set {
  public:
    // Lock on entire set 
    RW_lock *lock;

    FE_info_set(void);
    // creates a new set of FE_infos

    ~FE_info_set(void);

    FE_info *add_FE (fe_num const& id, FE_manager *mgr);
    // requires: this is write locked by the caller.
    //           mgr is FE manager for given fe_num.
    //           An entry for id isn't already present in this.
    // modifies: this
    // effects:  Creates a new FE_info for given FE, and returns it.
    //           Sets fe_num of FE_info to given id.
    
    bool remove_FE (fe_num const& id);
    // requires: this is write locked by the caller
    // modifies: this
    // effects:  If it is present, removes the entry for the given fe from
    //           the info set and returns TRUE; else returns FALSE.

    FE_manager *get_manager(fe_num const& id);
    // requires: this is read locked by the caller
    // effects:  Returns FE_manager associated with given FE, or 0 if none.

   // Generate all FE_infos in the set.
    class Elements {
      public:
        Elements(FE_info_set* set);
        // requires     set is not modified for the lifetime of generator.
	//              read or write lock is held during its lifetime
        // effects      Generate each element of set.  Each element is
        //              generated exactly once.

        bool get(FE_info*& fe_info);
        // modifies this, fe_tab
        // effects  If more infos are available, modify fe_info to
        //          contain the pointer to the next info and
        //          return TRUE.  Else do not modify fe_info, and return
        //          FALSE.
      private:
        FE_info_array* data;      // The actual contents of the FE_tables
        int index;                // Next table in the array
    };

  private:
    friend Elements;
    FE_info_array* set;  // Table that actually has the set.

    void free_FE_info(FE_info *info);
    // effects: Free fields of info, and then free info itself.
    //          Caller should NOT use info anymore.

    int find(fe_num const& id);
    // effects: Return slot number of FE_info in set with given 
    //          FE number, or -1 if none.
};
#endif /* _FE_INFO_SET_H */
