// Copyright 1995 Barbara Liskov

/*
\section{Front-End Connection Management}

This module maintains the connection with a particular FE.  Each "FE_Manager"
runs as a separate thread.  The thread loops reading messages from the
TCP connection to the FE and replying to them.

\subsection{Todo}
\begin{itemize}
\item Rework interface so that the lifetime of the FE management
      is not tied to the lifetime of a TCP connection.
\item Concurrent access?
\item Handle protocol errors
\end{itemize}
*/

#ifndef _FE_MANAGER_H
#define _FE_MANAGER_H

#include "common/basic.h"
#include "common/oref.h"
#include "common/tid.h"
#include "thread.h"
#include "prefetch_info.h"
#include "fe_info_set.h"

#include "config/vdefs/PAGING.h"
#if PAGING
#include "common/page.h"
#endif

class Samples;
class FE_table;
class Network;
class MM_HandleList;
class Pref_set;
class Transaction;
class Mutex;
class Condition;
class Xrefs;
class Uids;

class fe_message;
class prefetch_hint;
class or_objdesc;
union OR_slot;

extern "C" {
    struct iovec;
}

// \subsection{FE Manager}

class FE_manager : public Thread {
  public:
    FE_manager(fe_num id, Network* net);
    // requires - net is a TCP connection to the FE.
    //		  id is identifier for FE.
    // modifies - fe_info_set
    // effects  - Create a thread that handles messages from the FE.
    //		  The thread is responsible for shutting down the
    //		  connection when appropriate.  The thread is
    //		  initially inactive and has to be started
    //		  by calling the start() method.
    //            Adds the this FE\'s fe_table to set of fe_tables

    ~FE_manager();
    // effects - Shutdown the FE_manager.

    FE_info* fe_info();
    // effects - Returns the per-FE information for this FE.

    fe_num id();
    // effects - Returns the FE number of the FE we're connected to.

    virtual void main();
    // effects - Handle messages from connection.

    // \subsection{Transaction commit result} 
    // In a multiple-OR transaction, the FE manager waits on the 
    // condition variable until phase 1 of the commit is complete.  
    // When a thread learns the result of the transaction, it should 
    // grab the mutex, set "result" and signal the condition variable.

    Condition *cond_commit;
    Mutex     *mutex_commit;
    int       result;

  private:
/*
\subsection{Internal Operations}

Most operations are private because they only need to be accessed from
the "FE_manager" thread.  */

    void fetch_root();
    // effects - Send root oref to FE.

    void fetch_objects(Oref, prefetch_hint const*, int);
    // effects - Send specified object(s) to FE.

#if PAGING
    void fetch_page(Oref);
    // requires - no modifications take place during computation.
    //            This is only an experimental facility.
    // effects - Send page including specified object to FE.
    //
#endif

    void handle_stat_message();
    // effects	Responds with OR performance numbers.  See "fe_or_msg.h"
    //		for the format and contents of the message sent back to the FE.

    void commit_trans (bool coordinator);
    // effects - Commits the indicated transaction and sends the appropriate
    //           message to the FE.  coordinator tells whether or not this
    //           OR is the coordinator for the transaction.

    void commit_single(Transaction *tx);
    // requires - This OR is the only participant in the given transaction.
    // effects -  Perform commit for the given transaction, and send 
    //            appropriate message to FE.

    void commit_multiple(Transaction *tx, bool coordinator);
    // requires - The transaction involves multiple ORs; coordinator tells
    //            whether this OR is the coordinator.
    // effects  - Perform commit for the given transaction, and send 
    //            appropriate message to FE.

    void commit_read_only(Transaction *tx);
    // requires - The transaction is read-only everywhere.
    // effects  - Perform commit for the given transaction, and send 
    //            appropriate message to FE.

    void finish_transaction(Tid const& tid, int result);
    // effects  - Send result of given transaction back to FE.
    //            For use only with multiple OR transactions.

    void send_commit(Xrefs *xrefs, Uids *uids); 
    // effects  - Send commit message to FE with given new objects.

    void send_abort(int msgtype);
    // effects  - Send abort message to FE.

    void send_invalidation_message_if_needed();
    // effects - Send any invalid object set information that has not
    //           been sent to this frontend yet

    void process_invalid_ack(ubits32 last_msg_seen);
    // effects - Process the invalidation ack message received by removing
    //           the relevant objects from the invalid object set

    int  wait_or_timeout(fe_message *msg, int msec);
    // modifies msg
    // effects  waits for a message from the FE or times out after msec
    //          milliseconds. Returns -1 there was an error on the
    //          network connection with this FE. Returns 0 if there was a
    //          timeout. Else decodes the received message from the network,
    //          puts it into msg and returns 1

/*
\subsection{Fetch Code}

Internal operations used by the fetching code
*/

    void send_objects();
    // effects  - Send objects in "list" to the FE.

#if PAGING
    void send_page(Page const* p);
    // requires - no modifications take place.  This is only an experimental
    //            facility.
    // effects - Send page to FE.
    //           No information about sent objects recorded in info.
#endif

    void send_chunk(int start, int number);
    // requires	  1 <= number <= max_objects (see below)
    //		  [start..start+number-1] are legal indices into "list".
    // effects  - Send objects[start..start+number-1] to the FE.

/* \subsection{Representation} */
    Network*  net;
    FE_info   *fe_inf;      // FE table + invalid objects
    fe_num    fe_id;        // FE number of FE we're connected to

/*
The following storage is used by the thread to handle messages.
We make it part of the rep to avoid unnecessary allocation.
*/

#if USE_SUBPAGE_PREFETCHING
    Prefetch_Info*      pinfo;          // Prefetched objects collected here
#else
    MM_HandleList*	list;		// Prefetched objects collected here
#endif
    int			max_objects;	// Max no. of objects to send at a shot
    or_objdesc*		desc;		// Array for sending descriptors
    iovec*		iov;		// Gather array for sending objects
    Samples*		fetch_count;	// Tracks number of objs sent per fetch
    int                 tno;		// Number of transactions received
    Pref_set*		prefetch_set;	// Set of recently prefetched objects

/*
\subsection{Rep Invariant}

\begin{itemize}
\item "net" is a TCP connection to the FE
\item "fetable" records FE session information such as fetched and invalid
objects 
\item "thread" processes messages from "net"
\item "list" contains undeleted handles
\item "max_objects > 0"
\item "desc" points to an array of length "max_objects".
\item "iov" points to an array of length "(2 * max_objects + 1)".
      When we send objects,
      \begin{center}
      \begin{tabular}{ll}
      "iov[0]"&		points to descriptor area\\
      "iov[i+2]"&	points to data area for "ith" object
      \end{tabular}
      \end{center}
\end{itemize}
*/
};

#endif /* _FE_MANAGER_H */
