// Copyright 1995 Barbara Liskov

/*
\section{FE table specification}
  This module is the header for an FE-table, which is a per-FE description of
  objects referenced at the FE.  
  
  An FE table consists of 2 parts:
\begin{enumerate}  
\item A list of segments used by the FE.  Each segment is marked as either
     coarse grain or fine grain
\item For each segment stored at fine grain, list of EVERY object used by
     the FE in this segment.
\end{enumerate}
     Although there is no read/write lock associated with the FE table itself,
     the FE table is included in an FE_info structure (see fe_tables.h)
     which contains a lock for each FE.  In order to call the methods in an
     FE table, the caller should have the appropriate lock in the FE_info
     structure.

*/

/*
\subsection{Abstraction function}

A(r) = 
\begin{itemize}
\item Table corresponds to FE with id r.id.
\item List of used segments is given by hash table r.used_segs.  
\item A segment is fine-grained iff the high-order 2 bits of its 
 number in the hash table are nonzero.
\item List of used elements for a given fine-grained segment s is
given by hash table r.fine_segs->fetch(s).
\end{itemize}


\subsection{Rep invariant}
\begin{enumerate}
\item No segment appears more than once in r.used_segs.
\item If a segment is listed as fine-grained in r.used_segs, then an
entry exists for it in r.fine_segs, and vice versa.
\item Every oref in r.fine_segs->fetch(s) has its segment portion equal to s.
\item For any given fine-grained segment s, no oref appears more than
once in r.fine_segs->fetch(s).
\end{enumerate}

*/

#ifndef _FE_TABLE_H
#define _FE_TABLE_H

#include "common/basic.h"
#include "common/oref.h"
#include "common/array.h"
#include "common/openhashset.h"
#include "common/openhashmap.h"
#include "common/fe_num.h"
#include "thread.h"

class Oref_set;

// \subsection{Fine segment representation}
// In the FE table, a segment is stored as a 32-bit number.  The lower 30
// bits are the segment number.  The upper 2 bits are used to tell if a
// segment is fine grain or coarse grain.
//
// If the number of bits in a segment changes, change BITS_PER_SEGMENT
// and the masks.


typedef bits32 Seg;
#define BITS_PER_SEGMENT 30
#define SEGMENT_MASK 0x3fffffff
#define GRANULARITY_MASK 0xc0000000

#define Make_Seg(segid, granularity) \
                ((segid) | ((granularity) << BITS_PER_SEGMENT))
#define Get_Segid(seg) ((seg) & SEGMENT_MASK)
#define Get_Granularity(seg) (((seg) & GRANULARITY_MASK) >> BITS_PER_SEGMENT)

enum { coarse = 0, fine = 1 };  // Values for granularity

#define SEG_EQUAL(s1, s2) (Get_Segid(s1) == Get_Segid(s2))
#define SEG_HASH(s) (hash_int(Get_Segid(s)))
    
declareOpenHashSet(Seg_set, Seg, SEG_HASH, SEG_EQUAL)


// Structure for objects used by fine-grain segments
declareOpenHashMap(Segmap, Seg, Oref_set *, SEG_HASH, SEG_EQUAL)

class FE_table {
public:

//   \subsection{General operations}
    
    FE_table(void);
    // effects  - creates a new FE_table.
    
    ~FE_table(void);
    // effects - Destroy FE_table.    
    
//   \subsection{Used segment list operations}
    
    void add_segment(int segid);
    // requires: caller has a write-lock on this
    // modifies: this
    // effects:  
    //     If segid is already stored in the table, do nothing.
    //     Otherwise, add segid to this's list of used segments
    //       at coarse grain.
    
    void set_segment_list(int count, int segments[]);
    // requires: segments has count elements, starting with 0.
    //           caller has a write-lock on this
    // modifies: this
    // effects:  Set this's list of used segments to be a copy of the
    //           given segments array.
    //           All segments stored at coarse grain.
    
    void reset_segment_list(void);
    // requires: caller has a write-lock on this
    // modifies: this
    // effects:  Remove all segments from this's list of used segments.
    
    bool is_object_used(Oref x);
    // requires: caller has a read or write-lock on this
    // Let S be x's segment.
    // effects:  If S is being stored at coarse grain, return TRUE.
    //           If S is being stored at fine grain, return TRUE iff x
    //             is on S's list of used objects.
    //           Otherwise return FALSE.
    
    void add_object(Oref x);
    // requires: caller has a write-lock on this
    // Let S be x's segment.
    // modifies: this
    // effects:  
    //        If S is being stored at coarse grain, do nothing.
    //        If S is being stored at fine grain, add x to the list of 
    //          used objects for S in this.
    //        Otherwise insert S into list of used segments, at coarse grain.

//  \subsection{Garbage collection operations}

    void store_fine_grain(int segid);
    // requires: caller has a write-lock on this
    // modifies: this
    // effects:  Begin storing information for segment segid at the level
    //           of individual objects.
    //           If segid isn't in the list of used segments, insert it.
    //           (Rationale: if we're about to GC, must store info on segment.)
    //           Mark segid as a fine grain segment.
    
    void store_coarse_grain(int segid);
    // requires: caller has a write-lock on this
    // modifies: this
    // effects: If segid isn't on the list of used segments, insert it.
    //          Mark segid as a coarse grain segment.
    
    bool is_fine_grain(int segid);
    // requires: caller has a read or write-lock on this
    // effects:  Return TRUE iff segment segid is being stored at fine grain.
 
    Oref_set *fine_grain_elements(int segid);
    // requires   Segment segid is being stored at fine grain.
    // effects    Returns set of orefs in segment segid.
    //            The set is newly allocated and should be freed by the caller.
    
private:	
    Seg_set *used_segs;        // List of used segments
    Segmap *fine_segs;         // Objects used by segments stored at fine
			       // grain
    void delete_fine_orefs(void);
};

#endif /* _FE_TABLE_H */
