// Copyright 1995 Barbara Liskov

/*
\section{Main Program}

This is the "main" routine for an OR.
If you modify the argument processing code here, please update the
corresponding documentation in "doc/HOW_TO_RUN".
*/

#include <stdio.h>
#include <stdlib.h>

#include "common/compat.h"
#include "common/fail.h"
#include "common/fe_or_msg.h"

#include "mm/disk.h"

#include "init.h"
#include "or_config.h"
#include "or.h"
#include "server.h"
#include "or_or_msg.h"

static OR_config* parse_arguments(int argc, char* argv[]);
    // requires	argc is length of argv.  argv[0] is program name.
    // effects	Creates or configuration from command line and returns it.
    //		The caller should delete the returned object when it is
    //		no longer needed.  Dies after printing an error message
    //		if the command line is not well-formed.

static void usage(char const* prog);
    // requires	prog is program name.
    // effects	Print usage message and die.

int
main(int argc, char* argv[]) {

    or = new OR;
    or->config = parse_arguments(argc, argv);

    if (or->config->init != 0)
	OR_init();
    else
	OR_recover();

    // Provide service to FEs
    serve_connections();

    return 0;
}

// \subsection{Argument parsing code}
//
// If you modify this code, please update the corresponding documentation in
// "doc/HOW_TO_RUN".

static OR_config* parse_arguments(int argc, char* argv[]) {
    OR_config* config		= new OR_config;
    config->fe_socket		= OR_DEFAULT_FE_PORT;
    config->or_socket		= OR_DEFAULT_OR_PORT;
    config->ornum		= 0;  // Illegal #; must be set on startup
    config->cache_size		= OR_DEFAULT_CACHE_SIZE;
    config->log_size		= OR_DEFAULT_LOG_SIZE;
    config->apply_threshold	= OR_DEFAULT_APPLY_PERCENT;
    config->target_percent	= OR_DEFAULT_TARGET_PERCENT;
    config->port_offset		= 0;
    config->debug_level		= 0;
    config->init		= new OR_init_config;
    config->init->size		= (200 * 1024 * 1024) >> DISK_UNIT_SHIFT;
    config->init->segsize	= 128;
    config->init->input_file    = 0;

    int have_db      = 0;
    int have_size    = 0;
    int have_segsize = 0;

    int opt;
    int number;
    char* endptr;
    while ((opt = getopt(argc, argv, "ip:s:S:D:g:l:")) != EOF) {
	switch (opt) {
	  case 'p':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number < 0)) usage(argv[0]);
	    config->port_offset = number;
	    config->fe_socket   = OR_DEFAULT_FE_PORT + number;
	    config->or_socket   = OR_DEFAULT_OR_PORT + number;
	    break;
	  case 'l':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->log_size = number * 1024;
	    break;
          case 'i':
            have_db = 1;
            break;
	  case 's':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->init->size = (number * 1024 * 1024) >> DISK_UNIT_SHIFT;
	    have_size = 1;
	    break;
	  case 'S':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->cache_size = number * 1024;
	    break;
	  case 'D':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->debug_level = number;
	    break;
	  case 'g':
	    number = strtol(optarg, &endptr, 0);
	    if ((endptr == optarg) || (number <= 0)) usage(argv[0]);
	    config->init->segsize = number;
	    have_segsize = 1;
	    break;
	  default:
	    usage(argv[0]);
	    break;
	}
    }

    if ((argc - optind) != 1) usage(argv[0]);
    config->disk = argv[optind];

    // Check for conflicting options
    if (!have_db && (have_size || have_segsize)) usage(argv[0]);

    if (!have_db) {
	delete config->init;
	config->init = 0;
    }

    return config;
}

static void usage(char const* p) {
    warn("Usage: %s [-i [init options]] [options] <file>", p);
    warn("  <file>               Name of OR backing store.");
    warn("");

    warn("  General options (default values are in []):");
    warn("    -p <port>          Should be a small integer      [0]");
    warn("    -l <logsize>       Max log size in kilobytes      [1 mbyte]");
    warn("    -S <cachesize>     Max cache size in kilobytes    [8 mbytes]");
    warn("    -D <debug_level>   Print debugging data           [off]");
    warn("");

    warn("  Initialize options (default values are in []):");
    warn("    -s <size>          Database size in megabytes     [20]");
    warn("    -g <segsize>       Segment size in disk sectors   [128]");
    warn("");

    fail("*** exit ***");
}

