// Copyright 1995 Barbara Liskov

// \section{Disk Format}

#ifndef _DFORMAT_H
#define _DFORMAT_H

// The following is used to find the UFS superblock address.
// We use a similar address for our superblock so that we can
// skip any disk labelling information.
#include <ufs/fs.h>

#include "common/basic.h"
#include "common/bits.h"
#include "common/oref.h"
#include "common/or_obj.h"
#include "disk.h"

// \subsection{Super-blocks}
//
// Each OR maintains a pair of super-blocks in disk with
// checksums and timestamps.   The super-block with the higher time-stamp
// is used at recovery time.
//
// \begin{center}
// \begin{tabular}{|l|p{3in}|}
//	\hline
//	"DISK_SB_SIZE"&		Super-block size (in disk units)\\
//	"DISK_SB_1"&		Address of first super-block\\
//	"DISK_SB_2"&		Address of second super-block\\
//	\hline
// \end{tabular}
// \end{center}

#define DISK_SB_SIZE	1

#define DISK_SB_1	(SBOFF >> DISK_UNIT_SHIFT)
#define DISK_SB_2	(DISK_SB_1 + 1)

// Magic number to identify OR super-block
#define DISK_SB_MAGIC	0xA93C1E47

// Maximum length of log-file name
#define DISK_MAX_LOGLEN	100

// The super-block contains enough information to allow OR initialization.
// \begin{center}
// \begin{tabular}{|l|p{3in}|}
//	\hline
//	magic&		Contains magic number that can be used during
//			OR initialization to check that this is actually
//			the persistent storage for an OR.\\
//	checksum&	Checksum is used during recovery to detect
//			a partially updated superblock.  This situation
//			arises when an OR crashes while updating the block.\\
//	timestamp&	Used to find the newer of the two super-blocks.\\
//	\hline
//	size&		Number of blocks in disk (includes superblocks).\\
//	segprefsize&	Preferred segment size.\\
//	segtable&	Disk range occupied by segment table.\\
//	root&		Root oref.\\
//	log&		Log file name\\
//	\hline
// \end{tabular}
// \end{center}

#define DISK_SB_HEADER							      \
    ubits32	magic;							      \
    ubits32	checksum;						      \
    ubits32	timestamp;						      \
    Disk_Count	size;							      \
    Disk_Count	segprefsize;						      \
    Disk_Range	segtable;						      \
    Oref	root;							      \
    char	log[DISK_MAX_LOGLEN+1]

struct Disk_Sb_Header {
    DISK_SB_HEADER;
};

// The super-block structure
struct Disk_SB {
    DISK_SB_HEADER;

    // Pad until the superblock occupies the right number of disk blocks.
    char pad[DISK_UNIT*DISK_SB_SIZE - sizeof(Disk_Sb_Header)];
};

// \subsection{Segment Table}
//
// The segment table is maintained as a two-level map formed out
// of segments.  The root of the map is a segment stored at a
// wellknown location on disk.  This root segment consists of
// one object which has some small integer info followed by
// an array of disk ranges for the second level map entries.
//
// Each second-level map entry is a segment with a single object.
// The second-level object has some integers followed by an
// array of disk ranges for the segments mapped by this 2nd-level
// entry.

// Format of the single object contained in the root segment:
// (A disk range is encoded into an "OR_slot" by putting the
// address in "value32[0]" and the count in "value32[1]".
struct Disk_RootObj {
    // This part matches an OR_obj
    OR_slot	header[OR_obj_headers];

    ubits32	max_segments;	// Maximum number of segments allowed in db
    ubits32	num_leaves;	// # of leaf entries
    ubits32	segs_per_leaf;	// # of entries per leaf segment
    ubits32	pad;
    OR_slot	ranges[1];	// Disk ranges for leaf segments
};

// Format of the single object contained in a leaf segment:
// (A disk range is encoded into an "OR_slot" by putting the
// address in "value32[0]" and the count in "value32[1]".
struct Disk_LeafObj {
    // This part matches an OR_obj
    OR_slot	header[OR_obj_headers];

    ubits32	first;		// Segment number of first mapped entry
    ubits32	count;		// Number of entries in here
    OR_slot	ranges[1];	// Disk ranges for mapped segments
};

typedef Disk_Range* Disk_SegTable;

// \subsection{Segments}
//
// Each segment is considered an array of "OR_slot"s (each
// "OR_slot" is an eight-byte quantity).
//
// The first few OR_slots in the segment consist various pieces of
// information described below.
//
// After these slots, we store the offset of each object in this
// segment.  The offsets are stored as a simple array indexed by
// object index.  Each offset is a two-byte quantity that names the
// "OR_slot" at which the current object starts.  We can fit four
// offsets in each "OR_slot".  The offset array therefore occupues
// "ceil(count/4)" slots.
//
// The objects are stored in the area of the segment after the
// offset array.  We try to maintain a free area between the
// offset array and the object storage area.  This allows us
// to grow both the offset array and the object storage area
// easily.  Note that the object storage area may have holes
// because of objects that shrink in size.  Periodic compaction
// of the segment can be used to eliminate these holes.

// Segment magic number
#define DISK_SEG_MAGIC	0xA93C1E4A

// Since offsets are 16 bit quantities, cannot have segments with
// more than 2^16 slots.
#define DISK_SEG_MAX_SIZE (sizeof(OR_slot) * (1 << 16))

// Maximum object index for a segment
#define DISK_SEG_MAX_INDEX ((1 << 16) - 1)

// Segment types.  "Normal" holds Thor universe objects.
// "Special" holds OR runtime objects (these are not in the Thor universe).

#define DISK_SEG_NORMAL		0
#define DISK_SEG_SPECIAL	1

struct Disk_Segment {
    // Initial "OR_slot"s
    ubits32 magic;	// Magic number
    ubits16 id;		// Segment id
    ubits16 type;	// Segment type
    ubits16 slots;	// Number of slots in entire segment
    ubits16 count;	// Number of entries in "offsets"
    ubits16 objects;	// Smallest slot index occupied by an object.
    ubits16 size;	// Current combined size of all objects.

    // No padding necessary to make "header" start at "OR_slot" bounday.
    //		ubits16 pad[xxx];

    ubits16 header[1];	// Array of object offsets.

    // Object data occupies the slots "objects..(slots-1)".
    // Slots before "objects" and after the "header" area are free.
};

extern bool Disk_ReadSuper(Disk* disk, Disk_SB* super);
// modifies - super
// effects  - Reads superblock from "disk" into "super".
//	      Returns true iff successful.

#endif /* _DFORMAT_H */
