// Copyright 1995 Barbara Liskov

// \section{Disk Interface}
//
// A disk abstraction provides routines to read and write disk blocks.
// Concurrent access to the disk is controlled by "mm->mutex".  All
// of the disk methods require that the caller is holding "mm->mutex".
// The read and write operations release the mutex for the duration
// of their actual interaction with the disk.  (This is done so that
// other threads do not get blocked out of the MM while the long disk
// operation is running.)

#ifndef _DISK_H
#define _DISK_H

#include "common/array.h"
#include "common/bits.h"
#include "common/basic.h"
#include "common/compat.h"
#include "common/or_stat.h"

// \subsection{Basic Definitions}
//
// \begin{center}
// \begin{tabular}{|l|p{3.2in}|}
//	\hline
//	"DISK_UNIT"&		Number of units in a disk block (smallest
//				addressible disk unit).\\
//	"DISK_UNIT_SHIFT"&	Amount by which an address should be
//				shifted to multiply by "DISK_UNIT".\\
//	"Disk_Address"&		Type of disk addresses.  These address
//				"DISK_UNIT"s, not bytes.\\
//	"Disk_Count"&		Disk block counts.\\
//	"Disk_Range"&		Disk block ranges.\\
//	"Disk_RangeList"&	Resizable array of ranges.\\
//	\hline
// \end{tabular}
// \end{center}

#define DISK_UNIT	512
#define DISK_UNIT_SHIFT	9

typedef ubits32 Disk_Address;
typedef ubits32 Disk_Count;

struct Disk_Range {
    Disk_Address  address;
    Disk_Count	  count;
};

// Round number of bytes up to be a multiple of "DISK_UNIT"
inline ubits32 round_to_disk_blocks(ubits32 bytes) {
    ubits32 blocks = (bytes + DISK_UNIT - 1) >> DISK_UNIT_SHIFT;
    return (blocks << DISK_UNIT_SHIFT);
}

struct or_stat;
declareArray(Disk_RangeList,Disk_Range)

// Types of different disk requests.
// These are used as indices into an array, and therefore the
// entries should be numbered sequentially starting at zero.
enum Disk_OpType {
    Disk_Read   = 0,		// Untyped read
    Disk_Write	= 1,		// Untyped write
    Disk_FRead  = 2,		// Fetch read
    Disk_IRead  = 3,		// Installation read
    Disk_CRead  = 4,		// Cleaner read
    Disk_FWrite = 5,		// Flusher write
    Disk_CWrite = 6,		// Cleaner write
    Disk_NumTypes = 7		// How many types of operations are there?
};

// \subsection{Disk Interface}
class Disk {
  public:
    // Creators

    static Disk* open(char const* name);
    // effects	Recover from disk identified by "name".
    //		Return the open disk.  Return 0 on error.

    static Disk* open_fd(int fd);
    // effects	Create disk around specified file descriptor.

    ~Disk();
    // effects	Shutdown disk cleanly

    bool read(void* buffer, Disk_Range range, Disk_OpType = Disk_FRead);
    // requires	"buffer" is large enough to hold data in "range".
    // modifies	"buffer"
    // effects	Read data identified by "range" into "buffer".
    //		Returns true iff successful.  Returns false on error.
    //
    //		It is a checked run-time error if the specified "range"
    //		does not lie entirely within the disk.
    //
    //		The global variable "errno" is set appropriately when
    //		this routine returns false.

    bool write(void const* buffer, Disk_Range range, Disk_OpType = Disk_Write);
    // requires	"buffer" is large enough to hold data in "range".
    // modifies	"buffer"
    // effects	Write data from "buffer" to disk range identified by "range".
    //		Returns true iff successful.  Returns false on error.
    //
    //		It is a checked run-time error if the specified "range"
    //		does not lie entirely within the disk.
    //
    //		The global variable "errno" is set appropriately when
    //		this routine returns false.

    void stat(or_stat& stat);
    // effects	Store disk performance info in "stat".

// \subsection{Representation}
  private:
    Disk(int fd);

    // Disk file descriptor
    int device;

    // Type of operation to perform
    enum Op { Read_Op, Write_Op };

    bool do_io(Op op, void* buffer, Disk_Range range, Disk_OpType);
    // effects	Perform the operation indicated by "op".
    //		Returns true iff successful.

    // Performance measurements
    or_st_disk	total;	// Total statistics
    or_st_disk*	stats;	// Array indexed by operation type
    int		active;	// Number of currently active operations
    or_st_time	start;	// Time active period started (if active)
    or_st_time	busy;	// Time disk has been busy so far
};

#endif /* _DISK_H */
