// Copyright 1995 Barbara Liskov

/*
\subsection{Installation Table}

Recently committed objects sit in the log for a while before they get
written back to the appropriate segments.  The "itable" provides a
mapping from orefs to these objects so that clients fetches can
observe the most recent versions of such objects.

Concurrent access to the "itable" is controlled by "mm->mutex".
I.e., callers must hold "mm->mutex" before calling any itable
operations.
*/

#ifndef _ITABLE_H
#define _ITABLE_H

// The itable maintains a map from orefs to "log record, object ptr".

#include "common/basic.h"
#include "common/oref.h"
#include "common/array.h"
#include "common/or_obj.h"

class Itable;
class IM_Set;
class Itable_Rep;
class OR_obj;
class Data_Log_Record;

// Pending modified object state.
// These object states are reference counted.
struct Itable_Mod {
  public:
    static Itable_Mod* alloc(Oref oref, Data_Log_Record* lr, OR_obj* object);
    // effects	Create modified object state for specified object.
    //		The initial reference count is zero.

    void ref();
    // effects	Increment reference count for object.

    void unref();
    // requires	Corresponding earlier call to "ref()"
    // effects	Decrement reference count.  Delete if zero.

    Oref oref();
    // effects	Return object id

    Data_Log_Record* record();
    // effects	Return log record that generate the modification

    OR_obj* object();
    // effects	Return pointer to internally stored object.
  private:
    friend Itable;

    // Constructor/destructor are private because of special allocation
    Itable_Mod();
    Itable_Mod(Itable_Mod const&);
    void operator = (Itable_Mod const&);
    ~Itable_Mod();

    // The following constructor is used to create a mod used to search
    // a hash table.
    Itable_Mod(Oref oref);

    Oref		id;		// Obect id
    int			pin;		// Pin count
    Data_Log_Record*	lr;		// Log record
    OR_slot		header[1];	// Object contents starts here
};

declareArray(Itable_Mods,Itable_Mod*)

class Itable {
  public:
    Itable();
    ~Itable();

    void enter(Oref oref, Data_Log_Record* record, OR_obj* obj);
    // modifies	this
    // effects	Create modified object state and enter mapping into itable.
    //		Removes any previous mapping for the same object.

    Itable_Mod* lookup(Oref oref);
    // effects	If the itable contains a mapping for "oref", return
    //		the corresponding object state, otherwise return 0.
    //		The state of the returned modification is guaranteed
    //		to remain valid only until the "mm->mutex" is released,
    //		unless the caller increments the reference count of
    //		the modification.

    void remove(Itable_Mod* mod);
    // requires	"mod" was extracted from the itable by "get_modifications"
    //		and has since been written out.
    // effects  Remove the modification from itable and undo any reservations
    //		for the modification.

    bool segment_needs(int s, Data_Log_Record* record);
    // effects	Returns true iff segment "s" has a pending modification
    //		that is stored in the log record named by "record".

    void get_modifications(int seg, Itable_Mods* array);
    // modifies	"array"
    // effects	For each pending modification to "seg", append
    //		the pending modification to "array".  The caller
    //		should call "remove" when done writing out the
    //		modifications.  The caller should also "unref"
    //		the modifications.  The "unref" should happen
    //		after the call to "remove".

    int  has_modifications(int seg);
    // effects: returns number of modifications in itable for segment seg. 

    void dump();
    // for debugging
  private:
    void kill_modification(Itable_Mod* mod);
    // requires	Modification "mod" is no longer necessary, either
    //		because it has been overridden by another modification
    //		to the same object, or because it has been removed from
    //		the itable because of a disk write.
    // effects	Remove any reservation, log space usage for the modification,
    //		and also decrement the reference count held on behalf of the
    //		itable.

    get_segment_info(int segnum, IM_Set*& set);
    // effects: if information for segnum is present in the Itable rep,
    //          put the info in set and return TRUE. Else return FALSE
    //          and leave set unchanged


    Itable_Rep* rep;
};

// Some operations are inlined for speed.
inline void Itable_Mod::ref() {
    pin++;
}

inline void Itable_Mod::unref() {
    assert(pin > 0);
    pin--;
    if (pin == 0)
	delete [] ((char*) this);
}

inline Oref Itable_Mod::oref() {
    return id;
}

inline Data_Log_Record* Itable_Mod::record() {
    return lr;
}

inline OR_obj* Itable_Mod::object() {
    return ((OR_obj*) &header[0]);
}

#endif /* _ITABLE_H */


