#include <stdlib.h>
#include <stdio.h>

#include "common/arraysort.h"
#include "common/intarray.h"
#include "common/orefs.h"
#include "or/or.h"
#include "or/thread.h"

#include "dformat.h"
#include "mm.h"
#include "scache.h"
#include "disk.h"

#include "lmm/frag.h"
#include "lmm/finfo.h"
#include "lmm/ldformat.h"

// Maximum # of segments
static const int max_segs = 1 << 16;

typedef double (*Evaluator)(SegInfo*, int);

SegInfo::SegInfo() : map() {
    regions = b2r(or->mm->super->size-1) + 1;
    ignore = 0;
    segs = new IntArray[regions];
    next_seg = 0;
    //evaluator = eval_cost_benefit;
    evaluator = eval_greedy;

    mtime = new long[max_segs];
    for (int segnum = 0; segnum < max_segs; segnum++)
	mtime[segnum] = -1;

    live = new int[regions];
    for (int r = 0; r < regions; r++)
	live[r] = 0;

    check();
}

int SegInfo::num_regions() {
    return regions;
}

int SegInfo::alloc_seg() {
    // Search for free segment (mtime < 0)
    while ((next_seg < max_segs) && (mtime[next_seg] >= 0))
	next_seg++;
    assert(next_seg < max_segs);

    int result = next_seg;
    mtime[next_seg] = 0;
    next_seg++;
    return result;
}

void SegInfo::add_seg(int r, int s) {
    segs[r].append(s);
}

void SegInfo::moved(Oref oref, int slots, FragRange f) {
    // Update live info
    // XXX Assuming objects are not changing in size.
    int new_region = s2r(f.address);
    live[new_region] += slots;
    assert(live[new_region] <= region_size/sizeof(OR_slot));

    FragRange oldf = map[oref];
    if (oldf.address != 0) {
	int old_region = s2r(oldf.address);
	live[old_region] -= slots;
	if (live[old_region] == 0)
	    clear_region(old_region);
	assert(live[old_region] >= 0);
    }

    map[oref] = f;
}

void SegInfo::clear_region(int rnum) {
    th_assert(live[rnum] == 0, "clearing non-empty region");

    // Invalidate any cache entries for this region.
    int segcount = segs[rnum].size();
    for (int i = 0; i < segcount; i++) {
	int fid = make_fid(segs[rnum][i], rnum);
	or->mm->cache->invalidate(fid);
    }

    segs[rnum].clear();
}

void SegInfo::modified(int seg, long t) {
    if (mtime[seg] < t) mtime[seg] = t;
}

void SegInfo::find_object(Oref obj, FragRange& f) {
    f = map[obj];
    assert(f.address != 0);
}

int SegInfo::find_region(Oref obj) {
    FragRange f = map[obj];
    assert(f.address != 0);
    return s2r(f.address);
}

bool SegInfo::pick_empty(int& r) {
    for (int i = 0; i < regions; i++) {
	if (live[i] == 0) {
	    r = i;
	    return TRUE;
	}
    }
    return FALSE;
}

int SegInfo::empties() {
    int count = 0;
    for (int i = 0; i < regions; i++) {
	if (live[i] == 0)
	    count++;
    }
    return count;
}

long SegInfo::age(int r) {
    assert((r >= 0) && (r < regions));

    // Scan the ages of segments with live data in "r"
    long result = 0;
    int num = segs[r].size();
    for (int i = 0; i < num; i++) {
	int s = segs[r][i];

	// If segment has not been modified recently, there is no
	// need to check whether or not any of the live data in "r"
	// belongs to "s".
	if (result >= mtime[s]) continue;

	// Search the objects in "s" to see if they belong to "r"
	int num_objects = map.max_index(s) + 1;

	for (int j = 0; j < num_objects; j++) {
	    Oref o;
	    OREF_SET(o,s,j);
	    if (s2r(map[o].address) == r) {
		result = mtime[s];
		break;
	    }
	}
    }

    return result;
}

void SegInfo::live_segments(int r, IntArray* segments) {
    assert((r >= 0) && (r < regions));

    // Scan the segments with live data in "r"
    int num = segs[r].size();

    for (int i = 0; i < num; i++) {
	int s = segs[r][i];

	// Search the objects in "s" to see if they belong to "r"
	int num_objects = map.max_index(s) + 1;

	for (int j = 0; j < num_objects; j++) {
	    Oref o;
	    OREF_SET(o,s,j);
	    if (s2r(map[o].address) == r) {
		segments->append(s);
		break;
	    }
	}
    }
}

void SegInfo::live_objects(int r, int s, orefs* objects) {
    assert((r >= 0) && (r < regions));

    // Search the objects in "s" to see if they belong to "r"
    int num_objects = map.max_index(s) + 1;

    for (int j = 0; j < num_objects; j++) {
	Oref o;
	OREF_SET(o,s,j);
	if (s2r(map[o].address) == r)
	    objects->append(o);
    }
}

int SegInfo::live_ocount(int r, int s) {
    assert((r >= 0) && (r < regions));

    // Search the objects in "s" to see if they belong to "r"
    int num_objects = map.max_index(s) + 1;

    int count = 0;
    for (int j = 0; j < num_objects; j++) {
	Oref o;
	OREF_SET(o,s,j);
	if (s2r(map[o].address) == r)
	    count++;
    }
    return count;
}

// XXX Global variable used during sorting.  Contains pre-computed
// goodness values for cleaning purposes.
static double* good = 0;
static long cur_time;

// The comparison routine
static int comparer(void const* p1, void const* p2) {
    int r1 = *((int const*) p1);
    int r2 = *((int const*) p2);


    // Return "good[r2] - good[r1]" because we want the best candidates to
    // come to the front of the array and the evaluator returns
    // a larger number for a better candidate.
    double result = good[r2] - good[r1];
    if (result < 0.0) return -1;
    if (result > 0.0) return +1;
    return 0;
}

void SegInfo::pick_regions(int n, IntArray* list) {
    IntArray rlist(regions);
    for (int r = 0; r < regions; r++)
	rlist.append(r);

    cur_time = or->mm->new_stamp();
    good = new double[regions];
    for (r = 0; r < regions; r++)
	good[r] = goodness(r);
    ArraySort(rlist, comparer);

    // Check sorted list
    for (int x = 1; x < rlist.size(); x++)
	assert(good[rlist[x-1]] >= good[rlist[x]]);

    delete [] good;

    // Extract the first "n" elements of "rlist"
    if (n < rlist.size())
	rlist.remove(rlist.size() - n);
    list->concat(rlist);
}

double SegInfo::goodness(int r) {
    // Do not pick completely empty/full regions
    if ((live[r] == 0) || (live[r]*sizeof(OR_slot) == region_size))
	return 0.0;

    return (*evaluator)(this, r);
}

double SegInfo::eval_greedy(SegInfo* info, int r) {
    return ((double) (region_size/sizeof(OR_slot) - info->live[r]));
}

double SegInfo::eval_cost_benefit(SegInfo* info, int r) {
    long age = info->age(r);
    long elapsed = cur_time - age;

    double a = (((double) elapsed) / 1000000.0);
    double u = (((double) info->live[r]) / ((double) region_size));
    double e = (((1.0 - u) * a) / (2.0 * u));

    //debug("age = %8.3lfms", ((double) (((double) age) / 1000.0)));
    //debug("elp = %8.3lfms", ((double) (((double) elapsed) / 1000.0)));
    //debug("value = (%8.3lfs, %5.3lf) %13.3lf", a, u, e);

    return e;
}

double SegInfo::eval_worst(SegInfo* info, int r) {
    return ((double) info->live[r]);
}

void SegInfo::resize_dspace(int p) {
    // Want "p" percent of available disk space to be used.
    // First get amount of data.
    int data = 0;
    for (int r = 0; r < regions; r++)
	data += live[r];
    data *= sizeof(OR_slot);

    // Number of disk regions that should be used to have "p" percent free.
    int use_bytes = ((data / p) * 100);
    int use_regions = (use_bytes + region_size - 1) >> region_shift;

    // Number of regions to ignore...
    ignore = regions - use_regions;
    if (ignore < 0) ignore = 0;

    fprintf(stderr, "ignoring %4d/%4d regions\n", ignore, regions);
}

void SegInfo::check() {
    // XXX Implement
}

// Oref -> FragRange map.

declareArray(FragRanges,FragRange)
implementArray(FragRanges,FragRange)

FragMap::FragMap() {
    rep = new FragRanges[max_segs];
}

FragMap::~FragMap() {
    delete [] rep;
}

FragRange& FragMap::operator [] (Oref o) {
    assert(OREF_SEGMENT(o) < max_segs);

    int index = OREF_INDEX(o);
    int segment = OREF_SEGMENT(o);
    if (index >= rep[segment].size()) {
	// Enlarge the range array
	FragRange junk;
	junk.address = 0;
	junk.size = 0;
	rep[segment].append(junk, index - rep[segment].size() + 1);
    }

    return rep[segment][index];
}

int FragMap::max_index(int seg) {
    return rep[seg].size() + 1;
}

void SegInfo::init() {
    // Just write out empty region structures...
    Ldisk_Region* r = make_region(0);

    // We only need to write out at most one block at the
    // beginning of the region.
    Disk_Range range;
    range.count = 1;

    or->mm->mutex->grab(); {
	// Just write out arbitrary junk at the last disk block to make
	// sure the underlying file is big enough.
	range.address = or->mm->super->size - 1;
	bool result = or->mm->disk->write(r, range);
	assert(result);

	for (int i = 0; i < regions; i++) {
	    range.address = r2b(i);
	    result = or->mm->disk->write(r, range);
	    assert(result);
	}

	// The last region we write out in full to make the backing store
	// big enough.
	range.address = r2b(regions-1);
	range.count   = region_size >> DISK_UNIT_SHIFT;
	result = or->mm->disk->write(r, range);
	assert(result);
    } or->mm->mutex->release();

    delete_region(r);
}

struct RegionInfo {
    int	    number;
    ubits32 stamp;
};
declareArray(RegionList,RegionInfo)
implementArray(RegionList,RegionInfo)

// Routine to sort regions by increasing modification time.
static int compare_age(void const* p1, void const* p2) {
    RegionInfo* r1 = ((RegionInfo*) p1);
    RegionInfo* r2 = ((RegionInfo*) p2);

    return (r1->stamp - r2->stamp);
}

void SegInfo::recover() {
    or->mm->mutex->grab(); {
	// Get region info
	RegionList list;
	get_region_info(regions, list);

	// Sort with oldest regions first.
	ArraySort(list, compare_age);

	// Process the regions oldest to youngest
	Ldisk_Region* r = make_region(0);
	for (int i = 0; i < regions; i++) {
	    Disk_Range range;
	    range.address = r2b(list[i].number);
	    range.count   = region_size >> DISK_UNIT_SHIFT;

	    fprintf(stderr, "recovering region %3d/%3d [#%3d]\n",
		    i+1, regions, list[i].number);
	    bool result = or->mm->disk->read(r, range);
	    th_assert(result, "could not read region contents for recovery"); 

	    recover_region(list[i].number, r);
	}
	delete_region(r);

	int size = 0;
	for (i = 0; i < regions; i++)
	    size += live[i];

	fprintf(stderr, "total amount of live data = %8.2lf Mb\n",
		((double) (size*sizeof(OR_slot))) / ((double) (1024*1024)));
    } or->mm->mutex->release();
}

void SegInfo::get_region_info(int regions, RegionList& list) {
    Ldisk_Region* r = (Ldisk_Region*) new char[DISK_UNIT];
    for (int i = 0; i < regions; i++) {
	Disk_Range range;
	range.address = r2b(i);
	range.count   = 1;

	bool result = or->mm->disk->read(r, range);
	th_assert(result, "could not read region header for recovery");

	RegionInfo info;
	info.number = i;
	info.stamp  = r->stamp;
	list.append(info);
    }
    delete [] ((char*) r);
}

void SegInfo::recover_region(int rnum, Ldisk_Region* r) {
    OR_slot* addr = r->buf;
    int start = r2s(rnum) + ((char*)addr - (char*)r) / sizeof(OR_slot);

    int segcount = r->frags;
    for (int i = 0; i < segcount; i++) {
	// Process next fragment
	Ldisk_Fragment* frag = (Ldisk_Fragment*) addr;
	FragRange f;
	f.address = start;
	f.size    = frag->slots;

	addr += frag->slots;
	start += frag->slots;

	add_seg(rnum, frag->id);
	modified(frag->id, frag->stamp);

	int objcount = frag->count;
	for (int j = 0; j < objcount; j++) {
	    Oref oref;
	    OREF_SET(oref,frag->id,frag->header[j].id);
	    OR_obj* obj = (OR_obj*) (((OR_slot*) frag)+frag->header[j].offset);
	    moved(oref, OR_obj_full_size(obj), f);
	}
    }
}
