#include <stdio.h>
#include <struct.h>
#include "common/arraysort.h"
#include "or/or.h"
#include "disk.h"
#include "itable.h"
#include "mm.h"

#include "lmm/frag.h"

// Total # of data slots available per region.
static const int region_data_slots =
    (region_size - fldoff(Ldisk_Region, buf)) / sizeof(OR_slot);

// Total # of slots taken up by a fragment header.  This does not
// include any object slots or header entries.
static const int frag_header_slots =
    (fldoff(Ldisk_Fragment, header[0]) / sizeof(OR_slot));

Fragment::Fragment(int fid, FragRange r) : CacheEntry(fid) {
    contents = 0;
    range = r;
}

Fragment::Fragment(int fid, FragRange r, Ldisk_Fragment* c) : CacheEntry(fid) {
    range = r;
    contents = (Ldisk_Fragment*) new OR_slot[r.size];
    memcpy(contents, c, r.size * sizeof(OR_slot));

    initialized();
}

Fragment::~Fragment() {
    if (contents != 0)
	delete [] ((OR_slot*) contents);
}

int Fragment::size() const {
    return range.size * sizeof(OR_slot);
}

OR_obj* Fragment::pin(int index) {
    fetch();

    OR_obj* result = find(index);
    if (result != 0) add_pin();
    return result;
}

void Fragment::unpin() {
    th_assert(!missing(), "unpinning object in missing fragment");
    remove_pin();
}

void Fragment::write_contents() {
    th_assert(0, "fragment should not be written out individually");
}

void Fragment::read_contents(bool iread) {
    Disk_OpType t = iread ? Disk_IRead : Disk_FRead;

    // Get block-aligned range
    Disk_Address d1 = s2b(range.address);
    Disk_Address d2 = s2b(range.address + range.size - 1);

    Disk_Range aligned;
    aligned.address = d1;
    aligned.count   = (d2 - d1) + 1;

    // Allocate temporary buffer into which fragment is read.
    int bslots = (aligned.count << DISK_UNIT_SHIFT) / sizeof(OR_slot);
    OR_slot* buffer = new OR_slot[bslots];
    bool result = or->mm->disk->read(buffer, aligned, t);
    th_assert(result, "Fragment fetch failed");

    // Copy into contents.
    int offset = range.address - b2s(d1);
    OR_slot* data = new OR_slot[range.size];
    memcpy(data, buffer+offset, range.size * sizeof(OR_slot));
    contents = (Ldisk_Fragment*) data;
    delete [] buffer;

    th_assert(contents->magic == FRAG_MAGIC, "bad fragment magic number");
    th_assert(contents->id == f2seg(id()), "bad fragment id");
    th_assert(contents->slots == range.size, "bad fragment size");
}

OR_obj* Fragment::find(int index) {
    // Binary search for "index"
    int left = 0;
    int right = contents->count-1;
    Ldisk_Entry* header = contents->header;

    while (left <= right) {
	int mid = (left + right) >> 1;
	int mid_id = header[mid].id;

	if (mid_id == index)
	    return (OR_obj*) (((OR_slot*) contents) + header[mid].offset);

	// Adjust the search
	if (mid_id < index)
	    left = mid + 1;
	else
	    right = mid - 1;
    }

    return 0;
}

// Routine for sorting modification list
static int sort_by_index(void const* p1, void const* p2) {
    Itable_Mod* m1 = *((Itable_Mod**) p1);
    Itable_Mod* m2 = *((Itable_Mod**) p2);

    return (OREF_INDEX(m1->oref()) - OREF_INDEX(m2->oref()));
}

Ldisk_Region* make_region(ubits32 stamp) {
    Ldisk_Region* r = (Ldisk_Region*) (new char[region_size]);

    r->magic = REGION_MAGIC;
    r->stamp = stamp;
    r->frags = 0;
    r->size  = 0;
    r->sum   = 0;

    return r;
}

void delete_region(Ldisk_Region* r) {
    delete [] ((char*) r);
}

bool add_fragment(Ldisk_Region* r, int seg, long stamp, Itable_Mods* mods) {
    // Get the combined sizes of the objects.
    int oslots = 0;
    int num = mods->size();
    for (int i = 0; i < num; i++)
	oslots += OR_obj_full_size(mods->slot(i)->object());

    // XXX This code assumes two entries per slot.
    int hslots = (num + 1) / 2;

    // Get the fragment header size.
    int fslots = frag_header_slots + hslots;

    if ((oslots + fslots + r->size) > region_data_slots)
	// Fragment does not fit
	return FALSE;

    // Get fragment pointer to appropriate place in the region
    Ldisk_Fragment* f = (Ldisk_Fragment*) (r->buf + r->size);
    r->frags++;
    r->size += fslots + oslots;

    // Initialize fragment header
    f->magic	= FRAG_MAGIC;	// XXX Use different number?
    f->id	= seg;
    f->slots	= fslots + oslots;
    f->count	= num;
    f->stamp	= stamp;

    // Make a sorted copy of the modification list.
    Itable_Mods list = *mods;
    ArraySort(list, sort_by_index);

    // Initialize the header table
    int offset = fslots;
    for (i = 0; i < num; i++) {
	f->header[i].id      = OREF_INDEX(list[i]->oref());
	f->header[i].offset  = offset;

	OR_obj* obj = list[i]->object();
	int slots = OR_obj_full_size(obj);

	memcpy((((OR_slot*) f) + offset), obj, slots * sizeof(OR_slot));
	offset += slots;
    }

    assert(offset == f->slots);
    return TRUE;
}
