// MM code that is specific to the log-structured disk layout policy.

#include <stdio.h>
#include "common/fail.h"
#include "common/intset.h"
#include "or/or.h"
#include "or/thread.h"

#include "mm.h"
#include "logrecord.h"
#include "handle.h"
#include "itable.h"
#include "rtable.h"

#include "lmm/finfo.h"
#include "lmm/frag.h"
#include "lmm/ldformat.h"

static SegInfo* seginfo;

MM_Handle* MM::read_object(Oref oref, bool fast) {
    // Get fragment that contains "oref"
    FragRange f;
    seginfo->find_object(oref, f);

    // Get fragment id
    int fid = make_fid(OREF_SEGMENT(oref), s2r(f.address));

    // Look in the cache
    Fragment* frag = (Fragment*) cache->lookup(fid);
    if (frag == 0) {
	if (fast) return 0;

	// Try to allocate cache space for the fragment and then retry.
	cache->alloc(fid, f.address*sizeof(OR_slot), f.size*sizeof(OR_slot));
	return retry;
    }

    if (frag->missing()) {
	if (fast) return 0;

	// Wait for the fragment read to finish and then retry.
	OR_obj* obj = frag->pin(OREF_INDEX(oref));
	if (obj != 0) frag->unpin();
	return retry;
    }

    // Fetch object
    OR_obj* obj = frag->pin(OREF_INDEX(oref));
    if (obj == 0) return 0;

    // Found it
    MM_Handle* result	= new MM_Handle;
    result->in_itable	= FALSE;
    result->oref_	= oref;
    result->obj_	= obj;
    result->entry	= frag;
    return result;
}

CacheEntry* MM::alloc_entry(int id, long address, long size) {
    FragRange f;
    f.address = address / sizeof(OR_slot);
    f.size    = size / sizeof(OR_slot);

    return (new Fragment(id, f));
}

Segment* MM::find_segment(int segnum, bool fast) {
    th_assert(0, "invalid call to MM::find_segment");
    return 0;
}

int MM::free_space() {
    // XXX We do not keep track of free space.  Just return a large number.
    return (1 << 27);
}

void MM::install_object(Oref oref, OR_obj* obj) {
    th_assert(0, "invalid call to MM::install_object");
}

void MM::forward_object(Oref oref, OR_obj* obj) {
    th_assert(0, "invalid call to MM::forward_object");
}

static Ldisk_Region* collect_modifications(Log_Index& low, Log_Index& high,
					   Itable_Mods& mods) {
    Ldisk_Region* r = make_region(or->mm->new_stamp());
    Log* log = or->log;

    low = log->low();
    high = low - 1;


    // Scan one full region worth of modifications
    IntSet processed, tmp;
    Itable_Mods seg_mods;
    while (1) {
	// Can we get the segments from log[high+1]?
	int index = high+1;
	Log_Record* rec = log->fetch(index);
	if ((rec == 0) || !log->is_installed(index)) {
	    // Either this record has not been installed yet, or it
	    // has been removed from the log.
	    break;
	}

	// Check what segments from this log record fit into the region.
	bool fits = TRUE;
	tmp.clear();
	rec->get_modified_segments(&tmp);

	int seg;
	IntSet::Elements e = &tmp;
	while (e.get(seg)) {
	    if (processed.contains(seg)) continue;

	    // This segment has not been processed yet
	    seg_mods.clear();
	    or->mm->itable->get_modifications(seg, &seg_mods);
	    if (! add_fragment(r, seg, r->stamp, &seg_mods)) {
		fits = FALSE;
		continue;
	    }

	    // The contents of this segment fit.
	    processed.insert(seg);
	    mods.concat(seg_mods);
	    seginfo->modified(seg, r->stamp);
	}
	if (!fits) break;

	// All segments for this log record fit.
	high = index;
    }

    return r;
}

static void move_objects(int region, Ldisk_Region* r) {
    OR_slot* addr = r->buf;
    int start = r2s(region) + ((char*)addr - (char*)r) / sizeof(OR_slot);

    int segcount = r->frags;
    for (int i = 0; i < segcount; i++) {
	// Process next fragment
	Ldisk_Fragment* frag = (Ldisk_Fragment*) addr;
	FragRange f;
	f.address = start;
	f.size    = frag->slots;

	addr += frag->slots;
	start += frag->slots;
	seginfo->add_seg(region, frag->id);

	// Allocate a cache entry for this fragment if possible
	int fid = make_fid(frag->id, region);
	Fragment* fentry = new Fragment(fid, f, frag);
	if (or->mm->cache->fast_enter(fid, fentry))
	    or->mm->cache->used(fentry);
	else
	    delete fentry;

	int objcount = frag->count;
	for (int j = 0; j < objcount; j++) {
	    Oref oref;
	    OREF_SET(oref,frag->id,frag->header[j].id);
	    OR_obj* obj = (OR_obj*) (((OR_slot*) frag)+frag->header[j].offset);
	    seginfo->moved(oref, OR_obj_full_size(obj), f);
	}
    }
}

void MM::clean_log() {
    Log_Index low, high;

    mutex->grab(); {
	seginfo->clean_disk();

	fprintf(stderr, "[%3d empty regions]\n", seginfo->empties());

	// Pick an empty region
	int region;
	bool result = seginfo->pick_empty(region);
	th_assert(result, "could not find empty region");

	// Create in-memory region with some pending modifications.
	Itable_Mods mods;
	Ldisk_Region* r = collect_modifications(low, high, mods);

	or->mm->frag_writes += r->frags;
	or->mm->frag_sizes  += (r->size * sizeof(OR_slot));

	Disk_Range range;
	range.address = r2b(region);
	range.count   = region_size >> DISK_UNIT_SHIFT;
	result = disk->write(r, range, Disk_FWrite);
	th_assert(result, "region write failed");

	// Now update the in-memory location structures
	move_objects(region, r);
	int num = mods.size();
	for (int i = 0; i < num; i++) {
	    Itable_Mod* m = mods[i];
	    itable->remove(m);
	    m->unref();
	}
	installs += num;

	delete_region(r);
    } mutex->release();

    or->log->applied(low, high);
}

void MM::collect_segments(IntArray& segments, Log_Index& low, Log_Index& high){
    th_assert(0, "invalid call to MM::collect_segments");
}

void MM::sort_segments(IntArray& segments) {
    th_assert(0, "invalid call to MM::sort_segments");
}

void MM::write_modifications(IntArray const& segments) {
    th_assert(0, "invalid call to MM::write_modifications");
}

bool MM::alloc_space(int slots, Oref& oref) {
    if (slots >= DISK_SEG_MAX_INDEX) {
	warn("object is too big for this OR: %d slots", slots);
	return FALSE;
    }

    int bytes = slots * sizeof(OR_slot);
    if (bytes >= big_threshold) {
	// Give object a segment of its own.  The segment has a little
	// extra space to account for headers, etc.
	int seg = seginfo->alloc_seg();
	rtable->init_seg(seg, 0, slots+8);

	if (! rtable->reserve_new(seg, slots, oref)) {
	    warn("could not reserve space for big object");
	    return FALSE;
	}

	return TRUE;
    }

    // Try in normal segments that are used for small objects
    while (TRUE) {
	if (new_arena < 0) {
	    // Get a new segment of the preferred size.
	    new_arena = seginfo->alloc_seg();
	    rtable->init_seg(new_arena, 0,
			     ((super->segprefsize << DISK_UNIT_SHIFT) /
			      sizeof(OR_slot)));
	}

	// Try to make a reservation in this segment
	if (rtable->reserve_new(new_arena, slots, oref))
	    return TRUE;

	// Does not fit in new arena - try using another segment
	new_arena = -1;
    }
}

void MM::resize_dspace(int p) {
    mutex->grab(); {
	seginfo->resize_dspace(p);
    } mutex->release();
}

void MM::create_seg_table() {
    segtable = 0;		// Not used
    seginfo = new SegInfo;
    seginfo->init();
}

void MM::recover_seg_table() {
    segtable = 0;		// Not used
    seginfo = new SegInfo;
    seginfo->recover();
}
