// Copyright 1995 Barbara Liskov

// \section{Log Implementation}
//
// The contents of the log are managed in a circular buffer.
// Usually the log should not exceed a certain number of entries
// and therefore the log should wrap around in the circular
// buffer as entries are added and removed from the log.
// However, if entries are not being deleted from the log for
// some reason, the circular buffer is grown as needed.

// The log is protected from concurrent access by a mutex.

#include <stdio.h>
#include "common/th_assert.h"

// XXX The apply process is too eager at the moment.  It is woken up
// very often.  Plus it only applies a record at a time.

#include "or/thread.h"
#include "or/or.h"
#include "or/or_config.h"

#include "log.h"
#include "logrecord.h"
#include "mm.h"

// Each log entry contains a pointer to the actual log record,
// and the log record state.
struct Log_Entry {
    Log_Record* record;			// The actual record
    int		installed : 1;		// Is record installed?
    int		applied   : 1;		// Is record applied?
};

// Routine to calculate percentage of a number
inline int percent(int num, int percent) {
    return (num * (((double) percent) / 100.0));
}

// \subsection{Log Threads}

typedef void (Log::*Log_Proc)();

class Log_Thread : public Thread {
  public:
    Log_Thread(Log* l, Log_Proc p);
    virtual void main();
  private:
    Log* log;
    Log_Proc proc;
};

Log_Thread::Log_Thread(Log* l, Log_Proc p) {
    log = l;
    proc = p;
}

void Log_Thread::main() {
    // All the work is done inside a "Log" method
    (log->*proc)();
}

/* \subsection{Exported Operations} */

Log::Log() {
    mutex = new Mutex;
    apply_wait = new Condition(mutex);
    flush_wait = new Condition(mutex);
    flushers   = new Condition(mutex);
    space_wait = new Condition(mutex);
    death_wait = new Condition(mutex);

    // Create a small array initially
    list = new Log_Entry[1];
    size = 1;

    logsize = 0;
    max_logsize = or->config->log_size;
    apply_start = percent(max_logsize, or->config->apply_threshold);
    apply_stop  = percent(max_logsize, or->config->target_percent);

    start = 0;
    count = 0;
    next_index = 0;
    next_flush = 0;

    draining = 0;
    dying = 0;
    drain_wait = new Condition(mutex);

    // Fork off the apply thread
    Thread* a = new Log_Thread(this, &Log::apply_thread);
    a->start();

    // Fork off the flush thread
    Thread* f = new Log_Thread(this, &Log::flush_thread);
    f->start();
}

Log::~Log() {
    delete [] list;
    delete apply_wait;
    delete flush_wait;
    delete drain_wait;
    delete death_wait;
    delete flushers;
    delete mutex;

    // XXX Figure out how to kill the apply/flush threads
}

void Log::resize(int bytes) {
    mutex->grab(); {
	max_logsize = bytes;
	apply_start = percent(max_logsize, or->config->apply_threshold);
	apply_stop  = percent(max_logsize, or->config->target_percent);

	if (ready_to_apply()) apply_wait->signal();
    } mutex->release();
}

Log_Index Log::append(Log_Record* rec, bool client) {
    Log_Index result;

    mutex->grab(); {
	while (client && dying) {
	    // Client threads are not allowed to append stuff while OR
	    // is shutting down.  Just put the thread to sleep.  It
	    // will die when the OR dies.
	    death_wait->wait();
	}

	int recsize = rec->size();
	while ((logsize + recsize) > max_logsize)
	    space_wait->wait();

	if (count >= size)
	    enlarge();

	count++;
	Log_Entry* e = &(list[(start + count - 1) % size]);
	e->record = rec;
	e->installed = 0;
	e->applied = 0;
	next_index++;
	logsize += rec->size();

	if (ready_to_flush()) flush_wait->signal();
	if (ready_to_apply()) apply_wait->signal();

	result = next_index-1;
    } mutex->release();

    return result;
}

Log_Index Log::high() const {
    Log_Index h;
    mutex->grab(); {
	h = next_index-1;
    } mutex->release();
    return h;
}

Log_Index Log::low() const {
    Log_Index l;
    mutex->grab(); {
	l = next_index - count;
    } mutex->release();
    return l;
}

bool Log::is_stable(Log_Index index) const {
    bool result;
    mutex->grab(); {
	result = (index < next_flush);
    } mutex->release();
    return result;
}

bool Log::is_installed(Log_Index index) const {
    bool result;
    mutex->grab(); {
	int pos = find(index);
	if (pos < 0) {
	    // Index is out of range.  Has been installed only if index
	    // is old as opposed to new.
	    result = (index < next_flush);
	}
	else {
	    result = list[pos].installed;
	}
    } mutex->release();
    return result;
}

Log_Record* Log::fetch(Log_Index i) const {
    Log_Record* result;

    mutex->grab(); {
	int index = find(i);
	if (index < 0) {
	    result = 0;
	}
	else {
	    result = list[index].record;
	}
    } mutex->release();

    return result;
}

void Log::flush(Log_Index i) {
    mutex->grab(); {
	while (i >= next_flush) {
	    // Wakeup flush thread if necessary
	    flush_wait->signal();

	    // Wait for flush thread to do something
	    flushers->wait();
	}
    } mutex->release();
}

void Log::installed(Log_Index i) {
    mutex->grab(); {
	int index = find(i);
	if ((index >= 0) && !list[index].installed) {
	    list[index].installed = 1;
	    if (ready_to_apply()) apply_wait->signal();
	}
    } mutex->release();
}

void Log::applied(Log_Index start_index, Log_Index finish_index) {
    mutex->grab(); {
	// Mark these records as applied
	for (int i = start_index; i <= finish_index; i++) {
	    int index = find(i);
	    if ((index >= 0) && !list[index].applied) {
		list[index].applied = 1;
	    }
	}

	// We now try to delete all applied records at the low end
	int removed = 0;
	int oldsize = logsize;
	while ((count > 0) && list[start].applied) {
	    logsize -= list[start].record->size();
	    delete list[start].record;
	    list[start].record = 0;
	    start = (start + 1) % size;
	    count--;
	    removed++;
	}

	if (removed > 0) {
	    if (or->config->debug_level > 0)
		fprintf(stderr, "log: %8d -> %8d bytes\n", oldsize, logsize);

	    th_assert(count >= 0, "log count is invalid after applying");

	    // Wake-up drainer if present
	    if (draining && (count == 0)) {
		draining = 0;
		drain_wait->broadcast();
	    }

	    // Wake-up threads waiting for log space
	    space_wait->broadcast();
	}
    } mutex->release();
}

void Log::drain() {
    flush(high());

    mutex->grab(); {
	while (count > 0) {
	    draining = 1;
	    apply_wait->signal();
	    drain_wait->wait();
	}
    } mutex->release();
}

void Log::shutdown() {
    fprintf(stderr, "Draining the log...\n");

    mutex->grab(); {
	dying = 1;
    } mutex->release();

    drain();

    fprintf(stderr, "Done.\n");
    exit(0);
}

int Log::current_size() const {
    int result;
    mutex->grab(); {
	result = logsize;
    } mutex->release();
    return result;
}

int Log::target_size() const {
    int result;
    mutex->grab(); {
	// If the log is being drained, then the target size is 0.
	result = (draining ? 0 : apply_stop);
    } mutex->release();
    return result;
}

void Log::shrink(int bytes) {
    mutex->grab(); {
	assert(logsize >= bytes);
	logsize -= bytes;
    } mutex->release();
}

/* \subsection{Internal Operations} */

int Log::find(Log_Index i) const {
    Log_Index l = next_index - count;
    Log_Index h = next_index - 1;

    if ((i < l) || (i > h)) {
	return -1;
    }
    else {
	return (start + (i - l)) % size;
    }
}

void Log::enlarge() {
    // Enlarge the buffer
    int new_size = size*2;
    Log_Entry* new_list = new Log_Entry[new_size];
    for (int i = 0; i < count; i++) {
	new_list[i] = list[(start + i) % size];
    }

    delete [] list;
    list = new_list;
    size = new_size;
    start = 0;
}

bool Log::ready_to_apply() {
    // First make sure at least something is ready to be applied
    Log_Index lowest = next_index - count;
    if (!((count > 0) && (lowest < next_flush) && list[start].installed))
	return 0;

    return (draining || (logsize >= apply_start));
}

bool Log::ready_to_flush() {
    // XXX Currently, just see if there is an unflushed record.

    // It might make sense to not get ready to flush until
    // we accumulate a bunch of log records, or somebody is
    // waiting for a flush.

    return ((count > 0) && (next_index > next_flush));
}

/* \subsection{Threads Attached to the Log} */

void Log::apply_thread() {
    while (1) {
	mutex->grab(); {
	    // Wait for something to apply
	    while (! ready_to_apply())
		apply_wait->wait();
	} mutex->release();

	or->mm->clean_log();
    }
}

void Log::flush_thread() {
    // XXX This thread just marks everything as flushed right away.
    //	   This means that the log really is not stable.

    mutex->grab(); {
	while (1) {
	    // Wait for something to flush
	    while (! ready_to_flush())
		flush_wait->wait();

	    // XXX Obviously, the following is broken.
	    next_flush = next_index;
	    if (ready_to_apply()) apply_wait->signal();

	    // Wakeup all threads waiting for a log record
	    // to get flushed.  Hopefully, not too many
	    // threads will be waiting for a flush.
	    // We might have to do something more complicated
	    // if this assumption is incorrect.
	    flushers->broadcast();
	}
    } mutex->release();
}
