// Copyright 1995 Barbara Liskov

/*
  \section{Log Record Implementation}

  This file contains the implementation of various log records.

  XXX _We should think about splitting the different log record
  implementations into separate files._
  */

#include "common/or_obj.h"
#include "common/device.h"
#include "common/mos.h"
#include "common/nos.h"
#include "common/intset.h"
#include "common/intarray.h"
#include "common/th_assert.h"
#include "common/or_index.h"

#include "or/or.h"
#include "or/or_config.h"
#include "or/thread.h"

#include "itable.h"
#include "logrecord.h"
#include "mm.h"
#include "rinfo.h"

/*  Abstract base class.  This does not do much of anything. */

Log_Record::Log_Record(Tid t) : tid_(t) {
}

Log_Record::~Log_Record() {
}

Tid Log_Record::tid() const {
    return tid_;
}

bool Log_Record::absorbed() const {
    return FALSE;
}

int Log_Record::size() {
    return sizeof(Log_Record);
}

bool Log_Record::encode(Device* dev) {
    return tid_.encode(dev);
}

bool Log_Record::decode(Device* dev) {
    return tid_.decode(dev);
}

/* Data records. */

Data_Log_Record::Data_Log_Record(Tid t, Rinfo* r) : Log_Record(t), rinf(r) {
    aborted = FALSE;
    pending = r->mos->count() + r->nos->count();
    current_size = rinf->rep_size();

    Oref oref;
    OR_obj* obj;

    segments = new IntArray;
    IntSet segset;

    Mos::Elements mgen = rinf->mos;
    while (mgen.get(oref, obj)) {
	int seg = OREF_SEGMENT(oref);
	if (! segset.contains(seg)) {
	    segset.insert(seg);
	    segments->append(seg);
	}
    }

    int new_count = rinf->olist.size();
    for (int i = 0; i < new_count; i++) {
	int seg = OREF_SEGMENT(rinf->olist[i]);
	if (! segset.contains(seg)) {
	    segset.insert(seg);
	    segments->append(seg);
	}
    }
}

Data_Log_Record::Data_Log_Record() : Log_Record(tid()) {
    Mos* mos = new Mos;
    Nos* nos = new Nos;
    rinf = new Rinfo(mos, nos);
    pending = 0;
    current_size = rinf->rep_size();
    segments = new IntArray;
}

Data_Log_Record::~Data_Log_Record() {
    assert(pending == 0);
    if (rinf != 0) delete rinf;
    delete segments;
}

int Data_Log_Record::size() {
    return (Log_Record::size() + current_size);
}

bool Data_Log_Record::encode(Device* dev) {
    assert(rinf != 0);
    return (Log_Record::encode(dev) && rinf->encode(dev));
}

bool Data_Log_Record::decode(Device* dev) {
    assert(rinf != 0);
    if (Log_Record::decode(dev) && rinf->decode(dev)) {
	current_size = rinf->rep_size();
	return TRUE;
    }
    return FALSE;
}

void Data_Log_Record::install() {
    // If transaction committed, put new objects in itable.  Otherwise,
    // cancel reservations.
    MM* mm = or->mm;
    
    if (aborted) {
	mm->cancel(rinf);
    } 
    else {
	Itable* itable = mm->itable;

	mm->mutex->grab(); {
	    // Enter modified objects
	    Oref oref;
	    OR_obj* obj;
	    Mos::Elements mgen = rinf->mos;
	    while (mgen.get(oref, obj)) {
		itable->enter(oref, this, obj);
	    }
	    
	    // Enter new objects
	    int index = 0;
	    Modset::Elements ngen = rinf->nos;
	    while (ngen.get(obj)) {
		itable->enter(rinf->olist[index], this, obj);
		index++;
	    }
	} mm->mutex->release();
    }

    delete rinf;
    rinf = 0;
}

bool Data_Log_Record::absorbed() const {
    return (pending == 0);
}

void Data_Log_Record::get_modified_segments(IntSet* set) {
    if (pending == 0) return;

    Itable* itable = or->mm->itable;
    int num = segments->size();

    for (int i = 0; i < num; i++) {
	// Double-check with itable before putting segment
	int seg = segments->slot(i);
	if (!set->contains(seg) && itable->segment_needs(seg, this))
	    set->insert(seg);
    }
}

void Data_Log_Record::cancel(Oref oref, OR_obj* obj) {
    // Just ignore which object it is and decrement the value of the
    // pending modification counter.  This implies that
    // "get_modified_segments" will generate some extra segments in
    // many cases, but we can always double-check with the itable to
    // see if there are any pending modifications to these segments.

    pending--;
    th_assert(pending >= 0, "too many cancellations of log record contents");

    int size = OR_obj_full_size(obj) * sizeof(OR_slot);
    current_size -= size;
    assert(current_size >= 0);
}

void Data_Log_Record::commit(Xrefs *xrefs, OR_Index *index) {
    rinf->mos->assign_xrefs(xrefs, index);
    rinf->nos->assign_xrefs(xrefs, index);
}

void Data_Log_Record::abort() {
    aborted = TRUE;
    pending = 0;
    delete rinf;
    rinf = 0;
}

/* Commit record -- for single-OR transactions only. */

Commit_Log_Record::Commit_Log_Record(Tid t, Global_Tstamp stable_threshold)
     : Log_Record(t), threshold(stable_threshold)
{
}

Commit_Log_Record::~Commit_Log_Record() {
}

int Commit_Log_Record::size() {
    return Log_Record::size() + sizeof(threshold);
}

bool Commit_Log_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) && threshold.encode(dev));
}

bool Commit_Log_Record::decode(Device* dev) {
    return (Log_Record::decode(dev) && threshold.decode(dev));
}

void Commit_Log_Record::install() {
}

void Commit_Log_Record::get_modified_segments(IntSet* seg) {
}

/* Abort record. */

Abort_Log_Record::Abort_Log_Record(Tid t) : Log_Record(t) {
}

Abort_Log_Record::~Abort_Log_Record() {
}

int Abort_Log_Record::size() {
    return Log_Record::size();
}

bool Abort_Log_Record::encode(Device* dev) {
    return Log_Record::encode(dev);
}

bool Abort_Log_Record::decode(Device* dev) {
    return Log_Record::decode(dev);
}

void Abort_Log_Record::install() {
}

void Abort_Log_Record::get_modified_segments(IntSet* seg) {
    // Nothing to do
}

/* Prepared record --- logged at participants. */

Prepared_Log_Record::Prepared_Log_Record(Tid t, OR_num coord, Rinfo *r, 
					 Global_Tstamp tstamp, Xrefs *new_xrefs)
    : Data_Log_Record(t, r) 
{
    threshold = tstamp;
    coordinator = coord;
    xrefs = *new_xrefs;
}

Prepared_Log_Record::Prepared_Log_Record() : Data_Log_Record() {
}

Prepared_Log_Record::~Prepared_Log_Record() {
}

int Prepared_Log_Record::size() {
    return (Data_Log_Record::size() +
	    sizeof(threshold) +
	    sizeof(coordinator) +
	    encode_size(&xrefs));
}

bool Prepared_Log_Record::encode(Device* dev) {
    return (Data_Log_Record::encode(dev) &&
	    threshold.encode(dev) &&
	    dev->send_ubits32(coordinator) &&
	    encode_xrefs(&xrefs, dev));
}

bool Prepared_Log_Record::decode(Device* dev) {
    return (Data_Log_Record::decode(dev) &&
	    threshold.decode(dev) &&
	    dev->recv_ubits32(&coordinator) &&
	    decode_xrefs(&xrefs, dev));
}

/* Committing record --- logged at coordinator. */

Committing_Log_Record::Committing_Log_Record(Tid t, Rinfo *r, 
		 Global_Tstamp tstamp, OR_set *parts, Xrefs *new_xrefs)
    : Data_Log_Record(t, r) 
{
    threshold = tstamp;
    participants = *parts;  // Make copy of set of participants
    xrefs = *new_xrefs;
}

Committing_Log_Record::~Committing_Log_Record() {
}

int Committing_Log_Record::size() {
    return (Data_Log_Record::size() +
	    sizeof(threshold) +
	    participants.encode_size() +
	    encode_size(&xrefs));
}

bool Committing_Log_Record::encode(Device* dev) {
    return (Data_Log_Record::encode(dev) &&
	    threshold.encode(dev) &&
	    participants.encode(dev) &&
	    encode_xrefs(&xrefs, dev));
}

bool Committing_Log_Record::decode(Device* dev) {
    return (Data_Log_Record::decode(dev) &&
	    threshold.decode(dev) &&
	    participants.decode(dev) &&
	    decode_xrefs(&xrefs, dev));
}

/* Committed record --- logged at coordinator and participants. */

Committed_Log_Record::Committed_Log_Record(Tid t, Xrefs *new_xrefs, 
					   OR_Index *index, OR_set *parts) 
    : Log_Record(t) 
{
    // Make a copy of xrefs, so that they aren't deleted from under us
    xrefs = *new_xrefs;
    participants = *parts;  // Make copy of set of participants
    or_index = index;
}

Committed_Log_Record::~Committed_Log_Record() {
    delete or_index;
}

int Committed_Log_Record::size() {
    return (Log_Record::size() +
	    encode_size(&xrefs) +
	    or_index->encode_size() +
	    participants.encode_size());
}

bool Committed_Log_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) &&
	    or_index->encode(dev) &&
	    encode_xrefs(&xrefs, dev) &&
	    participants.encode(dev));
}

bool Committed_Log_Record::decode(Device* dev) {
    return (Log_Record::decode(dev) &&
	    or_index->decode(dev) &&
	    decode_xrefs(&xrefs, dev) &&
	    participants.decode(dev));
}

void Committed_Log_Record::install() {}
void Committed_Log_Record::get_modified_segments(IntSet* seg) {}

/* Done record. */

Done_Log_Record::Done_Log_Record(Tid t) : Log_Record(t) {
}

Done_Log_Record::~Done_Log_Record() {
}

int Done_Log_Record::size() {
    return Log_Record::size();
}

bool Done_Log_Record::encode(Device* dev) {
    return Log_Record::encode(dev);
}

bool Done_Log_Record::decode(Device* dev) {
    return Log_Record::decode(dev);
}

void Done_Log_Record::install() {}

void Done_Log_Record::get_modified_segments(IntSet* seg) {}

/* Participant record --- holds participant's prepare record in
   coordinator log protocol */

Participant_Log_Record::Participant_Log_Record(Tid t, OR_num participant,
					       Prepared_Log_Record *record,
					       Log_Index lindex) : Log_Record(t)
{
    index = lindex;
    pr = record;
    or_num = participant;
}

Participant_Log_Record::~Participant_Log_Record() {
    delete pr;
}

int Participant_Log_Record::size() {
    return (Log_Record::size() +
	    sizeof(ubits32) +
	    sizeof(ubits32) +
	    pr->size());
}

bool Participant_Log_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) &&
	    dev->send_ubits32(or_num) &&
	    dev->send_ubits32(index) &&
	    pr->encode(dev));
}

bool Participant_Log_Record::decode(Device* dev) {
    pr = new Prepared_Log_Record;

    return (Log_Record::decode(dev) &&
	    dev->recv_ubits32(&or_num) &&
	    dev->recv_ubits32(&index) &&
	    pr->decode(dev));
}

void Participant_Log_Record::install() {}
void Participant_Log_Record::get_modified_segments(IntSet* seg) {}

/* Stamos record --- used in coordinator log protocol */

Stamos_Record::Stamos_Record(OR_num coord, bool add) : Log_Record(tid()){
    or_num = coord;
    action = add;
}

Stamos_Record::~Stamos_Record() {
}

int Stamos_Record::size() {
    return Log_Record::size() + sizeof(ubits32) + sizeof(ubits32);
}

bool Stamos_Record::encode(Device* dev) {
    return (Log_Record::encode(dev) &&
	    dev->send_ubits32(or_num) &&
	    dev->send_ubits32(action));
}
bool Stamos_Record::decode(Device* dev) {
    return (Log_Record::decode(dev) &&
	    dev->recv_ubits32(&or_num) &&
	    dev->recv_ubits32(&action));
}

void Stamos_Record::install() {}
void Stamos_Record::get_modified_segments(IntSet* seg) {}
