// Copyright 1995 Barbara Liskov

/*
  \section{Log Record Interface}

  A Log record contains information designed to be stored in the
  log.  The class "Log_Record" is an abstract class and no objects
  of that class can be created.  Instead, it has various actual
  subclasses - one for each type of log record.
*/

#ifndef _LOGRECORD_H
#define _LOGRECORD_H

#include "common/oref.h"
#include "common/tid.h"
#include "common/or_set.h"
#include "common/xrefs.h"
#include "log.h"

class Device;
class Rinfo;
class IntSet;
class IntArray;
class OR_obj;
class Mos;
class Nos;
class Xrefs;
class OR_Index;

class Log_Record {
  public:
    Log_Record(Tid);
    // effects	- Create a log record for the specified transaction.

    virtual ~Log_Record();

    Tid tid() const;
    // effects	- Return id of transaction to which this record belongs.

    virtual int size() = 0;
    // requires	 Size is examined either for un-installed records only,
    //		 or if the record is installed, then only by the writer thread.
    // effects - Return the size of the log record in bytes.
    //		 The result in current implementations is only approximate.
    //		 Therefore it should only be used to trigger asynchronous
    //		 activity like flushing and writing.

    virtual bool encode(Device*) = 0;
    // requires	 Record has not yet been installed

    virtual bool decode(Device*) = 0;
    // requires	 Record has not yet been installed

    virtual void install() = 0;
    // effects	- Install modifications contained in this log record
    //		  into the OR volatile state.  This routine may try
    //		  to grab mm->mutex.

    virtual void get_modified_segments(IntSet* set) = 0;
    // requires	- mm->mutex is held.
    // modifies	- set
    // effects	- For each pending modification in the log record,
    //		  inserts the corresponding segment into set.  May
    //		  insert some segments even if there are no pending
    //		  mods for that segment.

    virtual bool absorbed() const;
    // requires	- mm->mutex is held.
    // effects	- Returns true iff all modifications have been superseded
  private:
    Tid tid_;
};

/*
  \subsection{Data Record}

  This record is generated for a transaction commit and it contains a
  list of modified objects.  The modified object set is created by the
  caller and handed off to the data record at data record creation
  time.  This interface is motivated by the desire to avoid copying
  modified objects around at the OR.
  */

class Data_Log_Record : public Log_Record {
public:
    Data_Log_Record(Tid, Rinfo* r);
    // requires	- r is a heap-allocated reservation info structure.
    // effects	- Creates a data record consisting of the objects in r.
    //		  r will be deleted automatically by the data log record.

    Data_Log_Record();
    // effects  - Creates empty log record, to be filled in by decode.

    virtual ~Data_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
    virtual bool absorbed() const;

    virtual void cancel(Oref oref, OR_obj* object);
    // requires	"log->mutex" is not held
    // effects	Mark the specified object as being removed from the record.
    //		Shrinks the log record size by an appropriate amt.

    virtual void abort();
    // effects  Mark this record as aborted, so that its changes 
    //          won't be installed.
    
    virtual void commit(Xrefs *xrefs, OR_Index *index);
    // effects  Modify MOS and NOS in log record to reflect new xrefs
    //          after transaction commits.  Also install all modified
    //		objects so that they woll be visible to fetch requests.
    
private:
    Rinfo*	rinf;		// The modified and new objects
    IntArray*	segments;	// Modified segment set
    int		pending;	// Number of pending object modifications
    int		current_size;	// Current log record size
    bool	aborted;	// true iff record was marked aborted
};


/*
  \subsection{Commit Record}

  This record is generated for a single-OR transaction commit and contains
  the threshold maintained by the transaction manager.
  */

class Commit_Log_Record : public Log_Record {
  public:
    Commit_Log_Record(Tid, Global_Tstamp stable_threshold); 
    // effects	- Creates a commit record consisting of the stable threshold.

    virtual ~Commit_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
  private:
    Global_Tstamp threshold;
};

/*
  \subsection{Abort Record}

  This record is generated when a transaction aborts.  The coordinator 
  logs it at soon as it learns that the transaction failed to validate
  at a participant; participants log it when they hear from the coordinator
  that the transaction has aborted.
  */

class Abort_Log_Record : public Log_Record {
  public:
    Abort_Log_Record(Tid);
    virtual ~Abort_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
};

/*
  \subsection{Prepared Record}

  This record is generated at a particpiant of a multiple-site transaction
  when the participant receives a prepare message and successfully validates.
  */

class Prepared_Log_Record : public Data_Log_Record {
public:
    Prepared_Log_Record(Tid, OR_num, Rinfo*, Global_Tstamp, Xrefs*);

    Prepared_Log_Record();
    // effects - create an empty log record, to be filled in by a call 
    //           to decode.

    virtual ~Prepared_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

private:
    Global_Tstamp threshold;
    OR_num coordinator;
    Xrefs xrefs;             // Xrefs of new objects
};

/*
  \subsection{Committing Record}

  This record is generated at the coordinator of a multiple-site transaction
  when the coordinator successfully validates the transaction.
  */

class Committing_Log_Record : public Data_Log_Record {
public:
    Committing_Log_Record(Tid, Rinfo *, Global_Tstamp, OR_set *, Xrefs*);
    virtual ~Committing_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

private:
    Global_Tstamp threshold;
    OR_set participants;
    Xrefs xrefs;             // Xrefs of new objects
};

/*
  \subsection{Committed Record}

  This record is generated at the coordinator and participants of a
  multiple-site transaction when the transaction successfully commits.

  We record the read-write participants in the transaction in the
  commit record at the coordinator; the coordinator is not included in
  this list of participants.  This field is empty in commit records at
  the participants.  
*/

class Committed_Log_Record : public Log_Record {
public:
    Committed_Log_Record(Tid, Xrefs *, OR_Index *index, OR_set *parts);
    virtual ~Committed_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
private:
    Xrefs xrefs;        // Xrefs of new objects
    OR_Index *or_index; // Mapping of participants to index in xrefs.
    OR_set   participants;  // Read-write participants in transaction.
};

/*
  \subsection{Done Record}

  This record is generated for a multiple-site transaction which has
  committed and for which the coordinator has received acknowledgments
  from all the participants.
  */

class Done_Log_Record : public Log_Record {
public:
    Done_Log_Record(Tid);
    // effects	- Creates a done log record for the given transaction

    virtual ~Done_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
};

/*
  \subsection{Participant Record}

  This is a prepare record we have received from a participant in the
  coordinator log protocol.  
*/
class Participant_Log_Record : public Log_Record {
public:
    Participant_Log_Record(Tid t, OR_num participant,
			   Prepared_Log_Record *record, Log_Index lindex);
    // effects	- Creates a participant log record for the given transaction

    virtual ~Participant_Log_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
private:
    OR_num    or_num;         // OR number of participant
    Prepared_Log_Record* pr;  // Participant's prepare record
    Log_Index index;     // Log index of participant's record *at participant*
};

/*
  \subsection{Stamos Record}

  Log an addition or deletion from the set of coordinators used in Stamos'
  optimization.
*/
class Stamos_Record : public Log_Record {
public:
    Stamos_Record(OR_num coord, bool add);
    // effects	- Creates a Stamos log record for the given coordinator.
    //            If "add" is true, this record is logging an addition
    //            to the set; otherwise it is a deletion.

    virtual ~Stamos_Record();

    virtual int  size();
    virtual bool encode(Device*);
    virtual bool decode(Device*);

    virtual void install();
    virtual void get_modified_segments(IntSet*);
private:
    ubits32   or_num;         // OR number of participant
    ubits32   action;         // true = add, false = delete
};



#endif /* _LOGRECORD_H */
