// Copyright 1995 Barbara Liskov

// \section{Memory Manager Implementation}

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "common/class_code.h"
#include "common/fail.h"
#include "common/fe_or_msg.h"
#include "common/obj_bitfield.h"
#include "common/or_obj.h"
#include "common/mos.h"
#include "common/nos.h"
#include "common/tid.h"
#include "common/th_assert.h"
#include "common/wk_xref.h"

#include "or/or.h"
#include "or/or_config.h"
#include "or/runtime.h"
#include "or/thread.h"

#include "dformat.h"
#include "handle.h"
#include "itable.h"
#include "rtable.h"
#include "log.h"
#include "logrecord.h"
#include "mm.h"
#include "rinfo.h"
#include "scache.h"
#include "segment.h"
#include "segtable.h"

#include "config/vdefs/PAGING.h"
#if PAGING
#include "pagetable.h"
#endif

MM::MM(bool init_flag) {
    th_assert(or->mm == 0, "duplicate MM creation");

    or->mm = this;
    mutex = new Mutex;
    new_arena = -1;
    cache = new Cache(or->config->cache_size);
    itable = new Itable;
    rtable = new Rtable;
    stamp_counter = 0;
    retry = new MM_Handle;

    mods	= 0;
    absorbed	= 0;
    installs	= 0;
    fetches	= 0;
    misses	= 0;
    frag_writes	= 0;
    frag_sizes	= 0;
    clean_count	= 0;
    clean_live	= 0;

    if (init_flag)
	init();
    else
	recover();

#if PAGING
    or->pages = new Page_Table(or, segtable);
#endif
}

/* \subsection{Reading Objects} */

MM_Handle* MM::fetch(Oref oref) {
    MM_Handle* h;
    mutex->grab(); {
	h = do_fetch(oref, FALSE);
    } mutex->release();
    return h;
}

MM_Handle* MM::fast_fetch(Oref oref) {
    MM_Handle* h;
    mutex->grab(); {
	h = do_fetch(oref, TRUE);
    } mutex->release();
    return h;
}

void MM::release(MM_Handle* h) {
    mutex->grab(); {
	delete h;
    } mutex->release();
}

MM_Handle* MM::do_fetch(Oref oref, bool fast) {
    // Remember the initial oref to place into the handle
    Oref orig = oref;

    // We need a loop to follow forwarding pointers.
    bool missed = FALSE;
    while (1) {
	Data_Log_Record* dr;
	OR_obj* obj;
	MM_Handle* result;

	// We first look for the named object in the "itable", and if
	// that does not succeed, we examine the cache.
	Itable_Mod* mod = itable->lookup(oref);
	if (mod != 0) {
	    result		= new MM_Handle;
	    result->in_itable	= TRUE;
	    result->oref_	= orig;
	    result->obj_	= mod->object();
	    result->mod		= mod;
	    mod->ref();
	}
	else {
	    // Object is not in the itable.  Look in the cache or on disk.
	    result = read_object(oref, fast);
	    if (result == retry) {
		missed = TRUE;
		continue;
	    }
	    if (result == 0) return 0;
	}

	// Follow forwarder?
	obj = result->obj();
	if (OR_OBJ_BITFIELD(obj) == OBJ_BF_MOVED) {
	    // Forwarding pointer
	    oref = OR_OBJ_FORWARDER(obj);
	    delete result;
	    continue;
	}

	// We increment miss count only at end to avoid multiple misses
	// for a single fetch.
	if (!fast) {
	    fetches++;
	    if (missed) misses++;
	}

	// Found it.  Make sure we report the requested oref.
	result->oref_ = orig;
	return result;
    }
}

Oref MM::directory() {
    MM_Handle* h = fetch(super->root);
    th_assert(h != 0, "could not fetch hidden OR root");
    Oref result = ((OR_hidden_root*) (h->obj()))->root.xref.oref;
    mutex->grab(); {
	delete h;
    } mutex->release();
    return result;
}

// \subsection{Modifications}

// Convert bytes to slots
inline b2s(int b) {
    return (b + sizeof(OR_slot) - 1) / sizeof(OR_slot);
}

bool MM::reserve(Rinfo* r) {
    bool result;

    mutex->grab(); {
	result = FALSE;

	// First get space reservation
	int slots = b2s(r->mos->obj_sizes() + r->nos->obj_sizes());
	if (rtable->reserve_space(slots)) {
	    // Get indices for new objects
	    if (reserve_nos(r))
		result = TRUE;
	    else
		rtable->remove(slots);
	}
    } mutex->release();

    return result;
}

/*
XXX _There should be some way to specify a preferred location for obj_

We currently just pass on the reservation request to the "new_arena"
segment.  The segment assigns an oref and reserves initial space for the
object.  We need a better mechansim for handling big objects here.
*/

bool MM::reserve_nos(Rinfo* r) {
    OR_obj* obj;
    Modset::Elements gen = r->nos;
    while (gen.get(obj)) {
	int slots = OR_obj_full_size(obj);
	assert(slots >= 0);

	Oref oref;
	if (! alloc_space(slots, oref)) {
	    // No way to clear the existing reservations
	    r->olist.clear();
	    return FALSE;
	}

	r->olist.append(oref);
    }
    return TRUE;
}

void MM::cancel(Rinfo* r) {
    mutex->grab(); {
	rtable->remove(b2s(r->mos->obj_sizes() + r->nos->obj_sizes()));

	// No way to cancel oref reservations...
	r->olist.clear();
    } mutex->release();
}

void MM::cancel_object(Oref oref, OR_obj* obj) {
    int slots = b2s(OR_obj_full_size(obj) * sizeof(OR_slot));
    rtable->remove(slots);

    // This object may be a new one and may have a pending reservation
    // in its segment.  However, we do not try to cancel that
    // reservation because this cancellation is probably caused by a
    // by another modification to this object and when this later
    // modification gets installed, the old reservation will get
    // cleaned up anyway.
}

long MM::new_stamp() {
    stamp_counter++;
    return stamp_counter;
}

void MM::stat(or_stat& stat) {
    mutex->grab(); {
	disk->stat(stat);
	stat.mods	  = mods;
	stat.absorbed	  = absorbed;
	stat.installs	  = installs;
	stat.logsize	  = or->log->current_size();
	stat.fetches	  = fetches;
	stat.misses	  = misses;
	stat.frag_writes  = frag_writes;
	stat.frag_sizes	  = frag_sizes;
	stat.clean_count  = clean_count;
	stat.clean_live   = clean_live;
    } mutex->release();
}

void MM::resize_cache(int bytes) {
    mutex->grab(); {
	cache->resize(bytes);
    } mutex->release();
}

void MM::resize_log(int bytes) {
    // Log has its own mutex.  No need to grab mm->mutex.
    or->log->resize(bytes);
}

// XXX Speed this up to avoid creating one log record per object?

Oref MM::make_object(OR_obj* obj) {
    Rinfo* rinfo = new Rinfo(new Mos, new Nos);
    Obj_Handle h = rinfo->nos->add_object(OR_OBJ_SIZE(obj));
    rinfo->nos->init_object(h,
			    OR_OBJ_CLASS(obj),
			    OR_OBJ_BITFIELD(obj),
			    OR_OBJ_SIZE(obj),
			    obj->slot);

    if (! reserve(rinfo))
	fail("could not reserve space for new object");

    // Peek into reservation strucuture for new oref
    assert(rinfo->olist.size() == 1);
    Oref result = rinfo->olist.high();

    // XXX Currently the TID value does not matter, so we do not initialize it
    Log_Record* rec = new Data_Log_Record(Tid(), rinfo);
    Log_Index index = or->log->append(rec);

    or->log->flush(index);
    rec->install();
    or->log->installed(index);

    return result;
}

void MM::modify_object(Oref oref, OR_obj* obj) {
    Rinfo* rinfo = new Rinfo(new Mos, new Nos);

    Obj_Handle h = rinfo->mos->add_object(oref, OR_OBJ_SIZE(obj));
    rinfo->mos->init_object(h,
			    OR_OBJ_CLASS(obj),
			    OR_OBJ_BITFIELD(obj),
			    OR_OBJ_SIZE(obj),
			    obj->slot);

    if (! reserve(rinfo))
	fail("could not reserve space for modified object");

    // XXX Currently the TID value does not matter, so we do not initialize it
    Log_Record* rec = new Data_Log_Record(Tid(), rinfo);
    Log_Index index = or->log->append(rec);

    or->log->flush(index);
    rec->install();
    or->log->installed(index);
}

/* \subsection{Initialization and Recovery} */


void MM::init() {
    // Create disk
    int fd = ::open(or->config->disk, O_RDWR|O_SYNC|O_CREAT, 0666);
    if (fd < 0) sysfail(or->config->disk);
    disk = Disk::open_fd(fd);

    // Check for disk vs. file
    struct stat prop;
    if (fstat(fd, &prop) < 0) sysfail(or->config->disk);
    th_assert((S_ISCHR(prop.st_mode) || S_ISREG(prop.st_mode)),
	      "specified file is neither a disk, nor a regular file");
    if (S_ISCHR(prop.st_mode)) {
	// Report actual statistics
	fprintf(stderr, "size    = %5d\n", prop.st_size);
	fprintf(stderr, "blksize = %5d\n", prop.st_blksize);
	fprintf(stderr, "blocks  = %5d\n", prop.st_blocks);

	// XXX _How do I get the actual disk size?_
    }

    // Create superblock

    super			= new Disk_SB;
    super->magic		= DISK_SB_MAGIC;
    super->checksum		= 0;
    super->timestamp		= 0;
    super->size			= or->config->init->size;
    super->segprefsize		= or->config->init->segsize;
    super->segtable.address	= 0;
    super->segtable.count	= 0;
    OREF_SET(super->root, 0, 0);
    sprintf(super->log, "logfile");

    // The following code tries to fit at least two non-big objects
    // into normal sized segments to avoid too much space loss
    // due to fragmentation.  We can achieve this by making
    // "big_threshold" be just less than half the preferred segment
    // size.

    int big_blocks = (super->segprefsize - 1) / 2;
    big_threshold = big_blocks << DISK_UNIT_SHIFT;

    create_seg_table();

    // Create hidden root for the OR
    super->root = make_root();

    // Now that the root object is known, write the superblocks to disk
    mutex->grab(); {
	Disk_Range r;
	r.count = DISK_SB_SIZE;
	r.address = DISK_SB_1;
	if (! disk->write(super, r)) sysfail(or->config->disk);
	r.address = DISK_SB_2;
	if (! disk->write(super, r)) sysfail(or->config->disk);
    } mutex->release();

    // Make sure everything is written out
    or->log->flush(or->log->high());
    or->log->drain();
}

void MM::recover() {
    disk = Disk::open(or->config->disk);
    if (disk == 0) sysfail(or->config->disk);

    mutex->grab(); {
	super = new Disk_SB;
	bool result = Disk_ReadSuper(disk, super);
	th_assert(result, "could not read disk superblock");

        // Set big threshold (See init() for details)
        int big_blocks = (super->segprefsize - 1) / 2;
        big_threshold = big_blocks << DISK_UNIT_SHIFT;

    } mutex->release();

    recover_seg_table();
}

// effects	Creates an object of the specified class,
//		length, and bitfield.
static OR_obj* make_obj(class_code cc, int length, Obj_bitfield bf) {
    OR_obj* x = ((OR_obj*) (new OR_slot[OR_obj_headers + length]));
    OR_OBJ_SIZE(x)		= length;
    OR_OBJ_CLASS(x)		= cc;
    OR_OBJ_BITFIELD(x)		= bf;
    return x;
}

// The code here depends on the rep of the directory class
// defined in "fe/types-th/orDirectory.th".

Oref MM::make_root() {
    Obj_bitfield bf;
    OR_obj* x;

    // Make an empty vector of nodes
    bf = 0;
    x = make_obj(WK_vec_OF_node_OREF, 0, bf);
    Oref vec = make_object(x);
    delete [] x;

    // Make an array of nodes that uses the preceding vector for its rep
    bf = 0;
    OBJ_BF_SETREF(bf, 0);
    OBJ_BF_SETDATA(bf, 1);
    OBJ_BF_SETDATA(bf, 2);
    x = make_obj(WK_array_OF_node_OREF, 3, bf);
    x->slot[0].xref.or     = or->config->ornum;
    x->slot[0].xref.oref   = vec;
    x->slot[1].value32[0]  = 0;
    x->slot[1].value32[1]  = 0;
    x->slot[2].value32[0]  = 0;
    x->slot[2].value32[1]  = 1;
    Oref rep = make_object(x);
    delete [] x;

    // Make the directory object that uses the preceding array as its rep
    bf = 0;
    OBJ_BF_SETREF(bf, 0);
    x = make_obj(WK_simple_directory_OREF, 1, bf);
    x->slot[0].xref.or     = or->config->ornum;
    x->slot[0].xref.oref   = rep;
    Oref dir = make_object(x);
    delete [] x;

    // Now make the real hidden root object
    bf = 0;
    OBJ_BF_SETREF(bf, 0);
    x = make_obj(0, 1, bf);
    x->slot[0].xref.or     = or->config->ornum;
    x->slot[0].xref.oref   = dir;
    Oref root = make_object(x);
    delete [] x;

    return root;
}
