// Copyright 1995 Barbara Liskov

/*
\section{Prefetching Implementation}

The following code gathers objects reachable from the prime object by
using a breadth-first search or simply gathering a subpage of the segment
arounf the object.

This code is bundled into the MM to avoid multiple mutex grabs/releases.
*/

#include "common/or_obj.h"
#include "common/objrefs.h"
#include "common/prefetch.h"
#include "common/map.h"
#include "or/thread.h"
#include "or/or.h"
#include "or/or_config.h"
#include "or/pref_set.t"
#include "or/prefetch_info.h"

#include "dformat.h"
#include "segtable.h"
#include "segment.h"
#include "mm.h"
#include "handle.h"
#include "itable.h"
#include "rtable.h"
#include "logrecord.h"

void MM::subpage_prefetch(Prefetch_Info* pinfo, Pref_set* pref_set,
			 struct prefetch_hint const* hint)  {

    int subpage_size = hint->max_prefetch;
    Oref oref = pinfo->first->oref();
    int segnum = OREF_SEGMENT(oref);
    int index = OREF_INDEX(oref);
    if (pinfo->no_objects == 0 || subpage_size == 0)
      return;
    int index_start = (index/subpage_size) * subpage_size;
    int index_end = index_start + subpage_size;
    Bitmap *b = 0;
    
  
    // Precompute how large the pinfo object set is allowed to get
    int max_no_objects = pinfo->no_objects + subpage_size;
  
    mutex->grab(); {
	// Lookup segment in the cache.
	Segment* seg = (Segment*) cache->lookup(segnum);

	// Check number of modifications for segment in mob.
	int nmodifications = itable->has_modifications(segnum);
    
	// Check if we recently sent any object in this segment to the FE.
	if (!pref_set->find(segnum, b)) {
	  // If not create a new entry for the segment
	  // with room for this subpage to fit in. 
	  b = new Bitmap(index_end);
	  pref_set->store(segnum, b);
	} else {
	  int min_bmap_size = (index_end + 31)/32;
	  if (b->size < min_bmap_size) {
	    // We need a larger bitmap for the segment.
	    // allocate it and copy the contents of the old bitmap 
	    // to the new one.
	    Bitmap *new_b = new Bitmap(index_end);
	    memcpy(new_b->bitmap, b->bitmap, b->size * sizeof(int)); 
	    pref_set->store(segnum, new_b);
	    delete b;
            b = new_b;
	  }
	}
	
	// Insert the object that was requested in the prefetch set.
	b->insert(index);

	// Note: We may have top pin the segment since the fetched object may
	// have been in the MOB
	bool segment_pinned = FALSE;
    
	for (int i = index_start; i < index_end; i++) {
	    bool found = FALSE;
	    OREF_SET(oref, segnum, i);
	    th_assert(sizeof(int)*8 == 32,
		      "Assumption sizeof(int) == 32 is false\n");
	    unsigned int div = i >> 5;
	    unsigned int mod = i - (div << 5);
	    
	    int *map_frag = b->bitmap+div;

	    if ((*map_frag >> mod) & 1) {
	      // If this object was recently sent to the FE do not prefetch it.
	      continue;
	    }
     
	    int no_objects = pinfo->no_objects;
	    Itable_Mod* modification = NULL;
	    if (nmodifications) {
		// We first look for the named object in the "itable", and if
		// that does not succeed, we examine the cache.
		modification = itable->lookup(oref);
		if (modification != 0) {
		    // The object was in the MOB.
		    pinfo->pref[no_objects].oref = oref;
		    pinfo->pref[no_objects].obj_ = modification->object();
		    pinfo->mod[pinfo->no_mods] = modification;
		    pinfo->no_mods++;
		    pinfo->no_objects++;
		    modification->ref();
		    found = TRUE;
                    nmodifications--;
		}
	    }

	    // If the object was not in the MOB and the segment is in the
            // cache, look for the object there.
	    if (!found && seg != 0) {
		OR_obj* obj = seg->find(i);
		if (obj != 0) {
		    // Found obj in the cache
		    found = TRUE;
		    pinfo->pref[no_objects].oref = oref;
		    pinfo->pref[no_objects].obj_ =  obj;
		    pinfo->no_objects++;
		    if (!segment_pinned) {
			segment_pinned = TRUE;
			seg->pin_segment();
			pinfo->entry = seg;
		    }
		}
	    }

	    // Could not find object 
	    if (!found) continue;

	    // We will not prefetch objects larger than DEFAULT_MAXSLOTS_OBJ
	    // or special objects.
	    OR_obj* obj = pinfo->pref[no_objects].obj_;
	    if (OR_OBJ_SIZE(obj) > DEFAULT_MAXSLOTS_OBJ ||
		OR_OBJ_CLASS(obj) == 0) {
		pinfo->no_objects--;
		if (modification)
		    pinfo->no_mods--;

		continue;
	    }
		
	    // Insert the object in the prefetch set.    
	    *map_frag |= ( 1 << mod); 
	    	    
	    // Do not send more than what was requested.
	    if (pinfo->no_objects >= max_no_objects)
		break;
	}
    } mutex->release();
    
    //fprintf(stderr, "SENDING %d\n", list->size());

    return;
}

void MM::pointer_prefetch(MM_HandleList* list, Pref_set* pref_set,
			 struct prefetch_hint const* hint)  {

    // Insert the object that was first requested in the prefetch set.
    pref_set->insert_oref(list->slot(0)->oref());

    // Precompute how large the list is allowed to get
    int max_list_size = list->size() + hint->max_prefetch;

    // Remember my OR number so that we do not follow cross-OR pointers
    int my_number = or->config->ornum;

    mutex->grab(); {
	int index = 0;
	while ((index < list->size()) && (list->size() < max_list_size)) {
	    // Prefetch based on orefs contained in list[index]
	    OR_obj* obj = list->slot(index)->obj();
	    index++;

	    int slotnum;
	    Object_References ref = obj;
	    while ((list->size() < max_list_size) && ref.get(slotnum)) {
		// Try to prefetch the object pointed to by the named slot
		Xref xref = obj->slot[slotnum].xref;
		if (xref.or != my_number) continue;
		Oref oref = xref.oref;
		if (pref_set->contains_oref(oref)) continue;

		MM_Handle* h = do_fetch(oref, TRUE);
		if (h == 0) continue;

		// Do not prefetch big objects
		if (OR_OBJ_SIZE(h->obj()) > DEFAULT_MAXSLOTS_OBJ) {
		    delete h;
		    continue;
		}

		list->append(h);
		pref_set->insert_oref(oref);
	    }
	}
    } mutex->release();

    //if (list->size() < max_list_size) {
    //	subpage_prefetch(list, orefs, hint);
    //}
}

void MM::release_list(MM_HandleList* list) {
    mutex->grab(); {
	int num = list->size();
	for (int i = 0; i < num; i++)
	    delete list->slot(i);
    } mutex->release();
}

void Prefetch_Info::clear(bool release_locks) {
    if (release_locks) {
	th_assert(first, "First is NULL in release_locks");
	delete first;
	if (entry)
	    entry->unpin();
	while (no_mods-- > 0)
	    mod[no_mods]->unref();
    }
    first = NULL;
    no_objects = 0;
    no_mods = 0;
    entry = NULL;
}

Prefetch_Info::Prefetch_Info(int nobjs) {
    clear(FALSE);
    pref = new prefetch_obj[nobjs];
    mod  = new Itable_Modp[nobjs];
}

Prefetch_Info::~Prefetch_Info() {
    delete [] pref;
    delete [] mod;
}
