// Copyright 1995 Barbara Liskov

/*
\section{Reservation Table Implementation}

The reservation table keeps track of information necessary to allow
reservations without having to read the corresponding segments into
the cache.  This works as follows:

\begin{itemize}

\item Store in volatile memory the information necessary to obtain
reservations.

\item The following per segment info should be sufficient: the amount
of free space, and an estimate of the free object indices.  This
information will be a conservative approximation of the segment
contents and will not be explicitly shadowed on disk.

\item At recovery time, we will just have an empty reservation table.
As segments are read into the cache, initialize the appropriate
entries in the rtable.  If the rtable becomes too full, just throw
away some of the entries.  Since we store very little info per
segment, this condition should not arise except for very large
databases with small segments.  If a reservation is required for a
segment without an rtable mapping, then we should either read the
segment to initialize the rtable info, or we should just satisfy the
reservation in some other "new_arena" segment.

\item Reservations for modified objects are made under the assumption
that either the object size has not changed, or if it has changed, the
object will still fit in the same segment.  We hedge our bets by
keeping a global free space counter and making space reservations
against that.  When such a modified object gets installed, we might
have to move the object to another segment because it will not fit in
its original segment.

\item Reservations for new objects are made by incrementing the
available index counter and by subtracting the appropriate amount of
space from the space counter specific to that segment.  We also make a
reservation against the global free space counter to maintain symmetry
with the modified-object reservation scheme.

\item If a transaction aborts or an object modification gets cancelled
because of write absorption, we have to undo the corresponding
reservations.  Global space reservations can be removed easily.  New
object reservations will not be removed.  Perhaps they can get fixed
up by the garbage collector, or periodically by scanning the log.

\end{itemize}
*/

#include "common/openhashmap.h"
#include "common/hashfuncs.h"
#include "common/or_obj.h"
#include "common/th_assert.h"
#include "or/or.h"

#include "rtable.h"
#include "dformat.h"
#include "mm.h"

// Info kept per segment
struct Rtable_SegInfo {
    ubits16	index;		// Next free index
    ubits32	space;		// Number of free bytes in segment
};

// Table mapping from segment number to segment info.
#define same_seg(a,b) ((a) == (b))
  declareOpenHashMap(Rtable_Map,int,Rtable_SegInfo,hash_int,same_seg)
implementOpenHashMap(Rtable_Map,int,Rtable_SegInfo,hash_int,same_seg)

struct Rtable_Rep {
    int		rslots;		// Total # of reserved slots
    Rtable_Map*	map;		// Segment number -> segment info map
};

Rtable::Rtable() {
    rep = new Rtable_Rep;
    rep->map = new Rtable_Map;
    rep->rslots = 0;
}

bool Rtable::reserve_space(int slots) {
    if ((rep->rslots+slots)*sizeof(OR_slot) <= or->mm->free_space()){
	rep->rslots += slots;
	return TRUE;
    }
    return FALSE;
}

bool Rtable::reserve_new(int segnum, int slots, Oref& result) {
    Rtable_SegInfo seg;
    if (! rep->map->fetch(segnum, seg)) {
	// XXX Should we read in the segment now?
	// Just return FALSE for now and let higher levels deal with it.
	return FALSE;
    }

    // See if index is available
    if (seg.index >= DISK_SEG_MAX_INDEX) {
	// No more indices available here
	return FALSE;
    }

    // Need extra space for header info as well.
    Disk_Segment s;
    int bytes = slots * sizeof(OR_slot) + sizeof(s.header[0]);
    if (seg.space < bytes) {
	// Not enough space here
	return FALSE;
    }

    // Make reservation
    OREF_SET(result,segnum,seg.index);
    seg.index++;
    seg.space -= bytes;
    rep->map->store(segnum, seg);

    return TRUE;
}

void Rtable::remove(int slots) {
    th_assert(slots <= rep->rslots, "cancelling unreserved slots");
    rep->rslots -= slots;
}

bool Rtable::initialized(int segnum) {
    return (rep->map->contains(segnum));
}

void Rtable::init_seg(int segnum, int index, int slots) {
    if (rep->map->contains(segnum)) return;
    if (slots > DISK_SEG_MAX_INDEX)
	slots = DISK_SEG_MAX_INDEX;

    Rtable_SegInfo seg;
    seg.index = index;
    seg.space = slots * sizeof(OR_slot);
    rep->map->store(segnum, seg);
}
