// Copyright 1995 Barbara Liskov

// \section{Server Cache}
//
// The server cache maintains a cache from ids to in-memory cache
// entries.  Access to the cache is controlled by the global
// "mm->mutex".  Callers should hold the mutex before calling any
// cache methods.

#ifndef _SCACHE_H
#define _SCACHE_H

#include "disk.h"
#include "common/link.h"

class  Cache_Table;
class  Condition;
class  CacheEntry;

class Cache {
  public:
    Cache(int size_in_bytes);
    // effects	Allocates cache with specified size.

    CacheEntry* lookup(int snum);
    // effects	Returns cache entry for "snum".  Returns 0 if
    //		not in cache.

    void invalidate(int snum);
    // effects	Invalidate any cache entry for "snum".

    bool alloc(int id, long address, long size);
    // requires	Cache has no entry for named cache entry.
    // effects	Attempt to allocate cache space for the specified
    //		entry.  Returns TRUE iff the mutex was not released
    //		while waiting for cache space.  If this routine
    //		returns TRUE, the cache is guaranteed to have an entry
    //		for "id".  Both address and size are specified in bytes.

    void enter(int snum, CacheEntry* entry);
    // requires	Cache has no entry for named entry.
    // effects	Enter "entry" into the cache.  This routine is typically
    //		used by the segment table code that has just allocated a
    //		segment.  This method may temporarily release "mm->mutex"
    //		while it waits for cache space to open up.

    bool fast_enter(int snum, CacheEntry* entry);
    // requires	Cache has no entry for named entry.
    // effects	If there is space in the cache, then it enters "entry"
    //		into the cache and returns TRUE.  Else it returns
    //		FALSE.  Unlike "enter", this method never releases
    //		"mm->mutex".

    void make_space();
    // requires	Some cached entry has become evictable.
    // effects	Allow threads waiting for cache space to run.

    void used(CacheEntry* entry);
    // requires	"entry" is in the cache
    // effects	Marks "entry" as used for the purposes of the cache replacement
    //		policy.

    void read_lock();

    void resize(int bytes);
    // effects	Resize cache to specified size.

    void dump();
  private:
    void evict(CacheEntry* entry);
    // effects	Evict the named entry.

    bool create_space(int entry_size);
    // effects	Attempt to create space for an entry with size "entry_size"
    //		by evicting any evictable entries.  Returns true iff
    //		successful.

    Cache_Table*	map;		// From ids to "Cache_Slot*"
    Link*		lru;		// LRU list head
    int			size;		// Current total size of cache entries
    int			max_size;	// Limit on cache size
    Condition*		space;		// To wait for cache space
};

// Cache entries have the following methods.
class CacheEntry {
  public:
    CacheEntry(int id);
    virtual ~CacheEntry();

    int id() const;
    // effects	Return cache index.

    virtual int size() const = 0;
    // effects	Return size (in bytes)

    virtual OR_obj* pin(int index) = 0;
    // effects	Pin object associated with specified index and
    //		return it. Return 0 if no such object exists.

    virtual void unpin() = 0;
    // requires	Object returned by an earlier pin(index) and has not
    //		been unpinned since.
    // effects	Unpin the object.

    bool missing() const;
    // effects	Returns true iff entry has not been initialized.

    bool evictable() const;
    // effects	Returns true iff entry can be safely evicted from memory.

    Link* lru_link();
    // effects	Return LRU link for this entry.  This routine is
    //		used by the cache to move the entry around within
    //		the cache LRU chain.

    void initialized();
    // requires	entry is absent.
    // modifies	this
    // effects	Mark this entry as clean and initialized.

    bool is_dirty() const;
    // effects	Returns true iff entry is dirty.

    void mark_dirty();
    // requires	!missing()
    // modifies	this
    // effects	Marks this as dirty.

    void write();
    // modifies	Disk representation of entry.
    // effects	All installed modifications are written to the disk image 

    void dump();
  protected:
    // Methods used only by subclasses.
    void add_pin();		// Increment pin count
    void remove_pin();		// Decrement pin count
    void fetch(bool iread = 0);	// Fetch from disk
    void write_lock();		// Wait until nobody is reading the entry.

    // The following are overridden by subclasses.
    virtual void read_contents(bool iread) = 0;
    virtual void write_contents() = 0;
  private:
    enum State { Absent, Reading, Clean, Dirty };

    Condition*	cond;		// Threads waiting for entry
    Link	lru;		// Link in the cache lru chain
    State	state;		// Current entry state
    int		cid;		// Cache id for this entry
    int		waiting;	// Number of threads waiting for entry
    int		pins;		// Current number of pending pins

    void release();
    // effects	Let other blocked threads proceed.
    // 		Call this routine whenever some thread has changed the
    //		entry state in a way that would allow some blocked
    //		thread to proceed.
};

inline int CacheEntry::id() const {
    return cid;
}

inline Link* CacheEntry::lru_link() {
    return &lru;
}

inline bool CacheEntry::is_dirty() const {
    return (state == Dirty);
}

#endif /* _SCACHE_H */
