// Copyright 1995 Barbara Liskov

/*
\section{Segment Implementation}

The representation of a segment consists of an array of "OR_slot"s.
The format of this array is given in "dformat.h".

\subsection{Rep Invariant}
XXX Fill this in

\subsection{Abstraction Function}
XXX Fill this in

\subsection{Todo}

\begin{itemize}
\item	Think about fairness for writers.  Currently, they can be
	continuously overtaken by readers.
\item	Interface to support multiple simulataneous ireads/writes.
\item	\ldots
\end{itemize}

*/

/*
\subsection{Assertion Checking}

The following code performs an assertion check and dumps out the
segment contents and aborts if the check fails.
*/
#define CHECK(cond) if (cond) {;} else { dump(); assert(cond); }

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "common/array.h"
#include "common/fail.h"
#include "common/or_obj.h"
#include "common/th_assert.h"
#include "or/or.h"
#include "or/thread.h"

#include "dformat.h"
#include "disk.h"
#include "mm.h"
#include "rtable.h"
#include "scache.h"
#include "segment.h"

/*
Header entries for existing object give the index in the segment at
which the corresponding object starts.  However, if the header entry
is "FREE_OBJECT", then no such object exists.
*/
#define FREE_OBJECT	0

// \subsection{Segment Creation}

Segment::Segment(int id, Disk_Range r) : CacheEntry(id) {
    th_assert(r.count >= 1, "segment must have at least one block");
    th_assert((r.count << DISK_UNIT_SHIFT) <= DISK_SEG_MAX_SIZE,
	      "segment is too big");

    range	= r;
    contents	= 0;
    write_stamp	= or->mm->new_stamp();
}

Segment::~Segment() {
    if (contents != 0) delete [] contents;
}

// \subsection{Segment Writer}
//
// The following special purpose thread is forked off whenever a
// new segment is created.  The job of this thread is to write out
// the new segment contents right away.  The segment contents have
// to be written out to prevent the cache from locking up with
// dirty new segments.  This has to be done in a separate thread to
// avoid releasing "mm->mutex" in the middle of segment initialization..
//
// XXX This is probably inefficient because the new segment will be
// written out again soon when it has been filled up with new objects.

class New_Segment_Writer : public Thread {
  public:
    New_Segment_Writer(Segment* seg);
    virtual void main();
  private:
    Segment* segment;
};

New_Segment_Writer::New_Segment_Writer(Segment* s) {
    segment = s;
}

void New_Segment_Writer::main() {
    or->mm->mutex->grab(); {
	segment->write();
    } or->mm->mutex->release();    
}

void Segment::init() {
    int slots = (range.count << DISK_UNIT_SHIFT) / sizeof(OR_slot);

    contents		= (Disk_Segment*) new OR_slot[slots];
    contents->magic	= DISK_SEG_MAGIC;
    contents->id	= id();
    contents->type	= DISK_SEG_NORMAL;
    contents->slots	= slots;
    contents->count	= 0;
    contents->objects	= slots;
    contents->size	= 0;
    write_stamp		= or->mm->new_stamp();

    initialized();
    mark_dirty();
    init_reservation_info();

    Thread* writer = new New_Segment_Writer(this);
    writer->start();
}

void Segment::read_contents(bool iread) {
    Disk_OpType t = iread ? Disk_IRead : Disk_FRead;

    int slots = (range.count << DISK_UNIT_SHIFT) / sizeof(OR_slot);
    contents = (Disk_Segment*) new OR_slot[slots];
    bool result = or->mm->disk->read(contents, range, t);
    th_assert(result, "Segment fetch failed");
    write_stamp = or->mm->new_stamp();
    init_reservation_info();
}

void Segment::write_contents() {
    write_stamp = or->mm->new_stamp();

    bool result = or->mm->disk->write(contents, range, Disk_FWrite);
    th_assert(result, "Segment write failed");
}

void Segment::init_reservation_info() {
    if (or->mm->rtable->initialized(id())) {
	// Do not bother trying to reinitialize
	return;
    }

    // Find the smallest index X such the all indices >= X are free.
    int free_index = contents->count;
    while ((free_index > 0) && (contents->header[free_index-1] == FREE_OBJECT))
	free_index--;

    // Tell the reservation table one less slot because of padding issues
    or->mm->rtable->init_seg(id(), free_index, free_slots() - 1);
}

Disk_Range Segment::disk_range() const {
    return range;
}

int Segment::size() const {
    return (range.count << DISK_UNIT_SHIFT);
}

bool Segment::modified(long stamp) const {
    return (is_dirty() && (write_stamp <= stamp));
}

void Segment::pin_segment() {
    or->mm->cache->used(this);
    add_pin();    
}

OR_obj* Segment::pin(int index) {
    fetch();

    OR_obj* result = find(index);
    if (result != 0) add_pin();
    return result;
}

void Segment::unpin() {
    remove_pin();
}

bool Segment::install(int index, OR_obj* object) {
    write_lock();

    // Get info about sizes of old and new versions
    int slots = OR_obj_full_size(object);
    OR_obj* old = find(index);
    int old_slots = ((old == 0) ? 0 : OR_obj_full_size(old));
    int delta = slots - old_slots;
    assert(slots >= 0);
    assert(old_slots >= 0);

    // Pre-allocate index if necessary
    if (index >= contents->count) {
	grow_header(index);
	if (index >= contents->count)
	    return FALSE;
    }

    // See if the object fits
    if (free_slots() < delta)
	return FALSE;

    mark_dirty();
    if ((old == 0) || (delta > 0)) {
	// Need to move object
	if (free_area_size() < slots) {
	    // Need to compact object storage
	    // compact();
	    fail("safe segment compaction has not been implemented\n");
	}

	assert(free_area_size() >= slots);
	contents->objects -= slots;
	contents->header[index] = contents->objects;
	old = find(index);
	assert(old != 0);
    }

    copy(object, old);
    contents->size += delta;

    return TRUE;
}

void Segment::grow_header(int index) {
    // How many extra header slots are needed.
    // We always allocate a full slot worth of look-up entries at a time (4).
    // Therefore, we want the new count to be the smallest multiple of 4 that
    // is greater than index.
    int new_count = ((index + 4) / 4) * 4;
    if (new_count <= contents->count) return;
    int extra_slots = (new_count - contents->count) / 4;

    // Check for space in the segment
    if (free_slots() < extra_slots) return;
    if (free_area_size() < extra_slots) {
	// Need to compact the object area.
	// compact();
	fail("segment compaction not implemented\n");
    }

    // Initialize the header up to the new count
    for (int i = contents->count; i < new_count; i++)
	contents->header[i] = FREE_OBJECT;
    contents->count = new_count;

    assert(free_slots()      >= 0);
    assert(free_area_size()  >= 0);
}

/*
\subsection{Forwarders}

We install a forwarder by overwriting the existing object.
*/

void Segment::install_forwarder(int index, Oref dest) {
    // Make a dummy forwarder object and install it
    OR_obj dummy;
    OR_OBJ_BITFIELD(&dummy) = OBJ_BF_MOVED;
    OR_OBJ_FORWARDER(&dummy) = dest;

    install(index, &dummy);
}

/*
\subsection{Internal Operations}

When these operations are called, the caller should already have
set-up things to prevent conflicting concurrent access to the segment.
These operations do not do any concurrency control.
*/

void Segment::copy(OR_obj* src, OR_obj* dest) {
    memcpy((void*)dest,
	   (const void*)src,
	   OR_obj_full_size(src)*sizeof(OR_slot));
}

OR_obj* Segment::find(int index) {
    if ((index < 0) || (index >= contents->count)) return 0;
    int offset = contents->header[index];

    return ((offset == FREE_OBJECT)
	    ? 0
	    : (OR_obj*) (((OR_slot*) contents) + offset)
	    );
}

int Segment::nobjects() {
  return contents->count;
}

int Segment::header_size() {
    // Two slots are taken up with miscellaneous information.
    // There are four entries per slot in the header.  Therefore
    // the lookup array occupies "ceil(header->count/4)" slots
    // and the entire header occupes two more than that.
    return (2 + (contents->count + 3) / 4);
}

int Segment::free_slots() {
    return (contents->slots - contents->size - header_size());
}

int Segment::free_area_size() {
    return (contents->objects - header_size());
}

int Segment::obj_area_size() {
    return (contents->slots - contents->objects);
}

void Segment::compact() {
    // Copy object area
    OR_slot* tmp = new OR_slot[obj_area_size()];
    memcpy(tmp, ((OR_slot*) contents)+contents->objects,
	   obj_area_size() * sizeof(OR_slot));

    int index = contents->slots;
    for (int i = 0; i < contents->count; i++) {
	if (contents->header[i] == FREE_OBJECT) continue;

	// Copy object from the tmp area back into segment
	OR_obj* o = (OR_obj*) (tmp + (contents->header[i]-contents->objects));
	index -= OR_obj_full_size(o);
	contents->header[i] = index;
	copy(o, find(i));
    }

    contents->objects = index;
    assert(obj_area_size() == contents->size);
}

void Segment::dump() {
    char const* f1 = "%-15s = %12d\n";
    char const* f2 = "%-15s = <%d,%d>\n";

    fprintf(stderr, f1, "segment",	id());
    fprintf(stderr, f2, "range",	range.address, range.count);

    if (contents != 0) {
	fprintf(stderr, f1, "slots",	contents->slots);
	fprintf(stderr, f1, "hsize",	header_size());
	fprintf(stderr, f1, "free",	free_slots());
	fprintf(stderr, f1, "farea",	free_area_size());
	fprintf(stderr, f1, "objarea",	obj_area_size());
    }
}
