// Copyright 1995 Barbara Liskov

/*
\section{Segment Interface}

A segment maps from an object index within a segment to
the corresponding object.
*/

#ifndef _SEGMENT_H
#define _SEGMENT_H

#include "common/basic.h"
#include "common/oref.h"
#include "disk.h"
#include "scache.h"

class Disk_Segment;
class OR_obj;
class Mutex;
class Condition;

class Segment : public CacheEntry {
  public:
// \subsection{Initialization}
    Segment(int segid, Disk_Range range);
    // requires	"range.count >= 1"
    // effects	Creates a segment for mapping objects within the
    //		segment specified by segid.  The segment is
    //		allowed to occupy the given "range".

    virtual ~Segment();
    // requires	"evictable()"
    // effects	Discard segment from memory.

    void init();
    // requires	"missing()"
    // effects	Initializes segment to contain no objects.
    //		After this call, the segment is no longer "missing".

// \subsection{Segment State}
    Disk_Range disk_range() const;
    // effects	Return disk range occupied by segment

    int size() const;
    // effects	Return segment size (in bytes)

    int nobjects();
    // requires	Caller holds "mutex" and segment is cached.
    // effects	Return number of objects in segment.

    bool modified(long stamp) const;
    // effects	- Returns true iff the segment has any pending modifications
    //		  made on or before "stamp".
    //
    //		  In typical use, a stamp will be generated when
    //		  installing a modification into a segment.  Later on
    //		  "modified(stamp)" can be called to check whether or
    //		  not that modification has been written to the disk.

// \subsection{Pinning Operations}

    OR_obj* pin(int index);
    void unpin();
    void pin_segment();

// \subsection{Object Modifications}

    bool install(int index, OR_obj* object);
    // modifies	this
    // requires	"index" is a valid object index into the segment.
    // effects	Attempts to store "object" value at the specified
    //		index.  Returns TRUE if successful, FALSE if the
    //		segment does not have enough room for the object.

    void install_forwarder(int index, Oref dest);
    // effects	Modify object named by "index" to forward to
    //		the object named by "dest".

    OR_obj* find(int i);
    // requires	Caller holds "mutex" and segment is cached.
    // effects	Returns the "i"th object within "contents".
    //		Returns 0 if no such object exists.

// \subsection{Hooks for CacheEntry}
  protected:
    void read_contents(bool iread);
    void write_contents();

// \subsection{Representation}
  private:
// Access to a segment is controlled by the MM mutex and a pin count.
    Disk_Range		range;		// Occupied disk range
    Disk_Segment*	contents;	// Actual contents (see "dformat.h")
    long		write_stamp;	// All mods before this stamp are
					// known to be on disk.

    void copy(OR_obj* src, OR_obj* dest);
    // requires	"dest" is big enough to hold "src".
    // modifies	"dest"
    // effects	Copies "src" into "dest".


    int header_size();
    // requires	Caller holds "mutex" and segment is cached.
    // effects	Return size of header area.  The header occupies
    //		slots "0..header_size()-1".

    
    int free_slots();
    // requires	Caller holds "mutex" and segment is cached.
    // effects	Return number of free slots in "contents".

    int free_area_size();
    // requires	Caller holds "mutex" and segment is cached.
    // effects	Return number of free slots between the header area and the
    //		object area.

    int obj_area_size();
    // requires	Caller holds "mutex" and segment is cached.
    // effects	Return number of slots in the object area.
    //		This will not necessarily be the same as "contents->size"
    //		because of gaps between objects.

    void grow_header(int index);
    // effects	Attempt to grow header to contain an entry for "index".
    //		May not succeed if segment does not have enough room.

    void compact();
    // requires	Caller holds "mutex", segment is cached and nobody is pinning
    // effects	Compacts the object storage area.

    void init_reservation_info();
    // requires	!missing()
    // effects	Initialize reservation info for this segment if necessary.

    void dump();
    // effects	Print segment rep to stderr
};

/* See "segment.cc" for rep invariant and abstraction functions. */

#endif /* _SEGMENT_H */
