// Copyright 1995 Barbara Liskov

/*
\section{OR Connection Management}

This module is the dual of the front-end connection manager (fe_manager)
for ORs.  Each OR_manager runs as a separate thread, reading messages
from the TCP connection and handling them.

The OR_managers are kept in an OR_manager set so that they can be
referenced later.  In order that all OR-OR communication be handled by
the OR_manager, all writes to an OR are funneled through an OR_manager
connected to that OR.  The OR_manager set is responsible for starting a 
new OR_manager if one doesn't already exist for the target OR of the write.

*/

#ifndef _OR_MANAGER_H
#define _OR_MANAGER_H

#include "common/basic.h"
#include "common/bits.h"
#include "common/openhashmap.h"
#include "common/or_num.h"
#include "common/locator.h"
#include "thread.h"
#include "or_or_msg.h"

class Network;

class OR_manager : public Thread {
  public:

    // constructor from existing connection
    OR_manager(OR_num id, Network* net);
    // requires - net is a TCP connection to the OR.
    //		  id is identifier for OR.
    // effects  - Create a thread that handles messages from the OR.
    //		  The thread is responsible for shutting down the
    //		  connection when appropriate.  The thread is
    //		  initially inactive and has to be started
    //		  by calling the start() method.

    ~OR_manager();
    // effects - Shutdown the OR_manager.

    virtual void main();
    // effects - Handle messages from connection.

    bool send_message(or_or_message const *msg);
    // effects - Send message to OR.  Returns TRUE iff successful.

  private:
/*
\subsection{Internal Operations}

These message handlers are only called from the main OR manager loop.
*/
    int  wait_or_timeout(or_or_message *msg, int msec);
    // modifies msg
    // effects  waits for a message from the OR or times out after msec
    //          milliseconds. Returns -1 there was an error on the
    //          network connection with this OR. Returns 0 if there was a
    //          timeout. Else decodes the received message from the network,
    //          puts it into msg and returns 1

    void vote_abort(Tid const& tid);
    // effects  Handle an abort vote from a participant for given transaction.

    void vote_ok(Tid const& tid, Xrefs *xrefs, Uids *uids);
    // effects  Handle an OK vote from a participant for given transaction.
    //          Xrefs and uids are for new objects at participant, or 0
    //          if transaction is read-only at participant.

    bool inform_fe(Tid const& tid, int result);
    // effects  If an FE manager exists for given transaction, inform 
    //          the manager of the given result and return true.
    //          Otherwise return false.

    void commit_trans(or_or_message *msg);
    // effects  Commit given transaction, as directed by coordinator.

    void abort_trans(Tid const& tid);
    // effects  Abort given transaction, as directed by coordinator.

    void ack(Tid const& tid);
    // effects  Handle an acknowledgment for given transaction.

/* \subsection{Representation} */
    OR_num    or_num;
    Network*  net;
    Mutex*    mutex_comm;   // Exclusive access to network during writes

/*
\subsection{Rep Invariant}

\begin{itemize}
\item "net" is a TCP connection to the OR
\item "or_num" is the OR number of the OR to which net is connected
\end{itemize}
*/
};

/*
  \subsection{OR Manager Set}
  The OR manager set's modification operations are protected by a mutex
  to allow concurrent updates.
  */

declareOpenHashMap(ORManagerTable, OR_num, OR_manager *, ORNUM_HASH, ORNUM_EQUAL)

class OR_manager_set {
public:
    OR_manager_set();
    ~OR_manager_set();

    bool send_message(OR_num or_num, or_or_message *msg);
    // effects - sends given message to given OR.  Starts new
    //           OR_manager if one isn't connected to or_num.
    //           Returns TRUE iff message sent successfully.
    
    void add(OR_num or_num, OR_manager *mgr);
    // effects - Set mgr to be the OR_manager for OR or_num.

    void remove(OR_num or_num);
    // effects - Set OR or_num to have no OR_manager.

private:
    ORManagerTable *table;
    Mutex *mutex;
};

#endif /* _OR_MANAGER_H */
