// Copyright 1995 Barbara Liskov

#include "common/network.h"
#include "common/xrefs.h"
#include "common/uids.h"
#include "common/or_index.h"
#include "mm/logrecord.h"
#include "or_or_msg.h"

// These encode/decode methods are rather complex.  The reason is so
// that we can just pass in an or_or_msg to an OR and have it send
// the entire message for us, even when the message is of variable
// length.  Otherwise sending a message to an OR would involve extracting
// the network connection from the OR, and ensuring atomic actions on messages
// would be more difficult.

bool or_or_message::encode(Network* net) const {
    struct iovec iov[2];
    or_or_vote_ok const *v;
    
    if (!net->send_ubits32(msgtype) || !tid.encode(net))
	return FALSE;

    switch(msgtype) {
    case OR_VOTE_OK_CL:
	// Send prepare record for coordinator log protocol
	if (!u.vote_ok.pr->encode(net) || !net->send_ubits32(u.vote_ok.index))
	    return FALSE;
	// fall through
    case OR_VOTE_OK:
	v = &u.vote_ok;

	// Encode the xrefs and uids
	iov[0].iov_base = (caddr_t) v->xrefs->as_pointer();
	iov[0].iov_len  = sizeof(Xref) * v->count;
	iov[1].iov_base = (caddr_t) v->uids->as_pointer();
	iov[1].iov_len  = sizeof(OR_uid) * v->count;

	return (net->send_ubits32(v->count) &&
	    net->send_vector(iov, 2));
	
    case OR_COMMIT:
	or_or_commit const *c = &u.commit;

	if (!c->index->encode(net))
	    return FALSE;

	// Encode the xrefs and uids
	iov[0].iov_base = (caddr_t) c->xrefs->as_pointer();
	iov[0].iov_len  = sizeof(Xref) * c->count;
	iov[1].iov_base = (caddr_t) c->uids->as_pointer();
	iov[1].iov_len  = sizeof(OR_uid) * c->count;
	
	return (net->send_ubits32(c->count) &&
	    net->send_vector(iov, 2));
    }
}

bool or_or_message::decode(Network* net) {
    ubits32 count;
    struct iovec iov[2];

    if (!net->recv_ubits32(&msgtype) || !tid.decode(net))
	return FALSE;

    switch(msgtype) {
    case OR_VOTE_OK_CL:
	// Get prepare record for coordinator log protocol
	u.vote_ok.pr = new Prepared_Log_Record();
	if (!u.vote_ok.pr->decode(net) || 
	    !net->recv_ubits32((ubits32 *) &u.vote_ok.index))
	    return FALSE;

	// fall through
    case OR_VOTE_OK:
	if (!net->recv_ubits32(&count))
	    return FALSE;
	u.vote_ok.count = count;
	u.vote_ok.xrefs = new Xrefs(count);
	u.vote_ok.uids  = new Uids(count);
	
	// Set up an iov to read in the two arrays
	iov[0].iov_base = (caddr_t) u.vote_ok.xrefs->as_pointer();
	iov[0].iov_len  = sizeof(Xref) * count;
	iov[1].iov_base = (caddr_t) u.vote_ok.uids->as_pointer();
	iov[1].iov_len  = sizeof(OR_uid) * count;
	
	return (net->recv_vector(iov, 2));

    case OR_COMMIT:
	u.commit.index = new OR_Index;
	if (!u.commit.index->decode(net))
	    return FALSE;

	if (!net->recv_ubits32(&count))
	    return FALSE;
	u.commit.count = count;
	u.commit.xrefs = new Xrefs(count);
	u.commit.uids  = new Uids(count);
	
	// Set up an iov to read in the two arrays
	iov[0].iov_base = (caddr_t) u.commit.xrefs->as_pointer();
	iov[0].iov_len  = sizeof(Xref) * count;
	iov[1].iov_base = (caddr_t) u.commit.uids->as_pointer();
	iov[1].iov_len  = sizeof(OR_uid) * count;
	
	return (net->recv_vector(iov, 2));
    }
}
