// Copyright 1995 Barbara Liskov

/*
\section{OR--OR Communication Protocol}

This file describes the OR--OR communication protocol.
The overall strategy is to set up a TCP connection between the
ORs and then send messages over that connection.

A connecting OR must first send its or_num, and then use the protocol
below.

*/

#ifndef _OR_OR_MSG_H
#define _OR_OR_MSG_H

#include "common/basic.h"
#include "common/bits.h"
#include "common/oref.h"
#include "common/xref.h"
#include "common/uid.h"
#include "common/tid.h"
#include "mm/logindex.h"

class Network;
class Xrefs;
class Uids;
class OR_Index;
class Prepared_Log_Record;

/*
OR's communicate with other OR's via a different port than with FEs.

OR ":0" gets assigned "OR_DEFAULT_OR_PORT".  OR $":"n$ gets assigned
"OR_DEFAULT_OR_PORT" + $n$.
*/
#define OR_DEFAULT_OR_PORT	9512

// \subsection{Constants for OR--OR messages} 

#define OR_COMMIT       302   /* Commit message in two-phase commit */
#define OR_ABORT        303   /* Abort message in two-phase commit */
#define OR_VOTE_OK      304   /* OK vote in two-phase commit */
#define OR_VOTE_ABORT   305   /* Abort vote in two-phase commit */
#define OR_VOTE_READ    306   /* Read-only vote in two-phase commit */
#define OR_ACK          307   /* Acknowledgment in two-phase commit */
#define OR_VOTE_OK_CL   309   /* OK vote in coordinator log protocol */

// \subsection{Possible OR to OR messages}

/* 
\subsubsection{The OR\_VOTE\_OK message}

This message has a header containing the number of new objects
made persistent by the transaction at an OR.  The header is followed
by two arrays.  One array is the list of xrefs assigned to
the new objects.  The second array is the list of uids assigned
to these new objects.
*/

struct or_or_vote_ok {
    ubits32	count;		// Number of new objects
    Xrefs*      xrefs;          // xrefs and uids of new objects
    Uids*       uids;
    // The following are for the coordinator log protocol only:
    Prepared_Log_Record *pr;    // Participant's prepare record
    Log_Index           index;  // Log index of prepare record
};

/* 
\subsubsection{The OR\_COMMIT message}

This message is similar to OR\_VOTE\_OK in that it gives xrefs and uids
of new objects.  However, it also includes an OR index which tells
which of the new xrefs come from which participants.
*/

struct or_or_commit {
    ubits32	count;		// Number of new objects
    Xrefs*      xrefs;          // xrefs and uids of new objects
    Uids*       uids;
    OR_Index*   index;          // Map from OR number to index in xrefs
};

// \subsubsection{The OR\_ABORT message}
// \subsubsection{The OR\_VOTE\_ABORT message}
// \subsubsection{The OR\_VOTE\_READ message}
// \subsubsection{The OR\_ACK message}
// These messages require no data beyond the transaction id.

struct or_or_message {
    ubits32 msgtype;
    Tid     tid;
    union {
	or_or_vote_ok     vote_ok;    // OR_VOTE_OK 
	or_or_commit      commit;     // OR_COMMIT
	// No additional data for ABORT, VOTE_ABORT, VOTE_READ, ACK messages
    } u;

    bool encode(Network*) const;
    bool decode(Network*);
};

#endif /* _OR_OR_MSG_H */
