// Copyright 1995 Barbara Liskov

/*
\section{OR Main Loop}

Loop waiting for connections from front-ends.  Each front-end connection
is allocated one thread to handle communication with that front-end.

\subsection{Todo}
\begin{itemize}
\item accept OR connections
      (need some way to distinguish them from FE connections)
\item recycle file descriptors if necessary
\item record "FE_manager" somewhere so that we can find the thread
      if necessary to handle a reconnection.
\end{itemize}
*/


#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include "common/other_unix.h"
#include "common/compat.h"
#include "common/fail.h"
#include "common/fe_or_msg.h"
#include "common/network.h"
#include "common/fe_num.h"

#include "fe_info_set.h"
#include "fe_manager.h"
#include "or_manager.h"
#include "or_config.h"
#include "or.h"
#include "server.h"
#include "or_or_msg.h"
#include "mm/log.h"

// Network error handler that prevents OR crashes.
static void print_warning(Device*, char const* msg) {
    warn(msg);
}

// Has a signal occurred?
// XXX Is there a race condition here?
static volatile int had_signal = 0;

#ifdef __osf__
void signalled_shutdown(int ignored) {
#else
void signalled_shutdown() {
#endif
    had_signal = 1;
}

void prepare_for_signals() {
    struct sigaction todo;

    // Shut down gracefully on SIGTERM
    todo.sa_handler = signalled_shutdown;
    sigemptyset(&todo.sa_mask);
    todo.sa_flags = 0;

    sigaction(SIGHUP,  &todo, 0);
    sigaction(SIGINT,  &todo, 0);
    sigaction(SIGTERM, &todo, 0);
}

void serve_connections() {
    or->fe_info_set = new FE_info_set; // Collection of per-FE information 
				       // (FE tables, etc.)

    // Handle complications introduced by signals
    prepare_for_signals();

    // Create sockets
    int fe_sock, or_sock;
    if ((fe_sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	sysfail("socket");
    }
    if ((or_sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	sysfail("socket");
    }

    // Allow address reuse
    int opt = 1;
    if (setsockopt(fe_sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt)) < 0) {
	sysfail("setsockopt");
    }
    if (setsockopt(or_sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt)) < 0) {
	sysfail("setsockopt");
    }


    // Bind address
    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(or->config->fe_socket);
    if (bind(fe_sock, (struct sockaddr*) &addr, sizeof(addr)) < 0) {
	sysfail("bind");
    }
    
    addr.sin_port = htons(or->config->or_socket);
    if (bind(or_sock, (struct sockaddr*) &addr, sizeof(addr)) < 0) {
	sysfail("bind");
    }
    

    // Allow connections
    if (listen(fe_sock, 5) < 0) {
	sysfail("listen");
    }
    if (listen(or_sock, 5) < 0) {
	sysfail("listen");
    }

    fprintf(stderr, "Server ready\n");

    // Accept connections
    while (1) {
	fd_set readfds;
	struct sockaddr_in from;
	int from_size = sizeof(from);
	int fd, retval, numfds;

	// Wait for FE or OR connection
	FD_ZERO(&readfds);
	FD_SET(fe_sock, &readfds);
	FD_SET(or_sock, &readfds);
	numfds = getdtablesize();

	if (select(numfds, &readfds, NULL, NULL, NULL) < 0) {
	    if ((errno == EINTR) && had_signal)
		break;
	    
	    syswarn("select");
	    continue;	    
	}

	// Check for new FE connection 
	if (FD_ISSET(fe_sock, &readfds)) {
	    if ((fd = accept(fe_sock, (struct sockaddr*) &from, &from_size)) < 0) {
		if ((errno == EINTR) && had_signal)
		    break;
		
		syswarn("accept");
		continue;
	    }
	    
	    Network* net = new Network(fd);
	    net->set_handler(print_warning);
	    net->set_nodelay();
            net->set_buffsizes();
	    
	    // Obtain FE identifier
	    fe_num id;
	    if (! id.decode(net)) {
		// XXX - Log error
		fprintf(stderr, "could not read fe identifier\n");
		delete net;
		continue;
	    }
	    
	    FE_manager* mgr = new FE_manager(id, net);
	    mgr->start();
	}

	// Check for new OR connection
	if (FD_ISSET(or_sock, &readfds)) {
	    if ((fd = accept(or_sock, (struct sockaddr*) &from, &from_size)) < 0) {
		if ((errno == EINTR) && had_signal)
		    break;
		
		syswarn("accept");
		continue;
	    }

	    Network* net = new Network(fd);
	    net->set_handler(print_warning);
	    net->set_nodelay();
            net->set_buffsizes();

	    // Read OR number
	    OR_num id;
	    if (! net->recv_buffer(&id, sizeof(id))) {
		// XXX - Log error
		fprintf(stderr, "could not read OR identifier\n");
		delete net;
		continue;
	    }

	    // Start new OR manager and add to manager set
	    fprintf(stderr, "Got new OR number %ld\n", id);
	    OR_manager* mgr = new OR_manager(id, net);
	    or->or_managers->add(id, mgr);
	    mgr->start();
	}
    }

    // Done processing connections because of a signal
    or->log->shutdown();
}
