// Copyright 1995 Barbara Liskov

#include "common/oref_set.h"
#include "common/or_obj.h"
#include "common/generator.h"

#include "fetch.h"
#include "server.h"

struct State {
    Server*	server;		// OR pointer
    ObjectMap*	objects;	// Objects already traversed
    int		fetches;	// Number of explicit fetch requests
    prefetch_hint* hint;	// Prefetching hints
};

static void fetch_proc(State* state, Oref oref);

static void report(State* state) {
    printf("traversed: %7d, fetches: %5d\n",
	   state->objects->size(),
	   state->fetches);
}

ObjectMap* fetch_all(Server* server, Oref oref, prefetch_hint* h) {
    State state;
    state.server = server;
    state.objects = new ObjectMap;
    state.hint = h;
    state.fetches = 0;

    fetch_proc(&state, oref);
    report(&state);

    return state.objects;
}

static void fetch_proc(State* state, Oref oref) {
    if (state->objects->contains(oref)) return;

    // Fetch the object
    ObjectMap* map = state->server->fetch(oref, state->hint);
    state->fetches++;

    // First enter all fetched objects into visited set so that
    // we do not try to fetch them again.

    Oref_set visit;
    for (ObjectMap::Bindings x = map; x.ok(); x.next()) {
	Oref o = x.key();
	OR_obj* v = x.val();

	// Do not visit an object more than once
	if (state->objects->contains(o)) {
	    delete [] ((OR_slot*) v);
	    continue;
	}

	// Stash away object information
	state->objects->store(o, v);
	if ((state->objects->size() % 1000) == 0) report(state);

	// Remember that we need to visit this object
	visit.insert(o);
    }

    // Now visit all the newly fetched objects

    Oref o;
    Generator<IntKey> *gen = visit.keys();
    IntKey ik;
    while (gen->get(ik)) {
        o = ik.val;
	OR_obj* obj = map->fetch(o);

	// Scan the references in this object
	Obj_bitfield bf = OR_OBJ_BITFIELD(obj);
	for (int i = 0; i < OR_OBJ_SIZE(obj); i++) {
	    if ((bf == OBJ_BF_ALLREFS) ||
		(!OBJ_BF_SPECIAL(bf) && OBJ_BF_ISREF(bf,i))) {
		fetch_proc(state, obj->slot[i].xref.oref);
	    }
	}
    }

    delete gen;

    delete map;
}
