// Copyright 1995 Barbara Liskov

/*
\section{Thread Interface}

This module provides the thread interface used at the OR.  The types
implemented here are "Thread", "Mutex", and "Condition".
*/

#ifndef _THREAD_H
#define _THREAD_H


#include "common/basic.h"

struct Mutex_Rep;
struct Condition_Rep;

class Mutex;
class Condition;

/*
\subsection{Thread}

A "Thread" object has a method called "main" that can be executed as
a separate thread of control.
*/

class Thread {
  public:
    Thread();
     /*
      * effects	Create a new thread.  The thread is initially inactive.
      */
     
    virtual ~Thread();
    /*
     * requires	Thread is not active.
     * effects	Deletes all associated storage.
     */
    
    void start();
    /*
     * effects	Execute main() as a separate thread.
     *		When main returns, delete thread.
     */

    virtual void main() = 0;
    /*
     * effects	Execute thread body.  This method is overridden by
     *		sub-classes to provide real thread bodies.
     */
};

/*
\subsection{Mutex}

A "Mutex" provides mutual exclusion to synchronize mutiple processes that
access shared state.
*/

class Mutex {
  public:
    Mutex();
    ~Mutex();

    void grab();
    // requires - this thread has not already grabbed the mutex
    void release();
    // requires - this thread has grabbed the mutex
  private:
    friend class Condition;
    Mutex_Rep* rep;
};

/*
Macro for holding a mutex while running some code.

Usage:
\begin{verbatim}
	GRAB(mutex, {
	    ...
	    C statements
	    ...
	});
\end{verbatim}
*/

#define GRAB(m, body) do { (m)->grab(); body; (m)->release(); } while (0)
    // requires - The C statements do not attempt to leave the body
    //		  except by falling of the end.  I.e. no returns, breaks,
    //		  continues, gotos out of the body.

/*
\subsection{Condition Variables}

A condition variable is another synchronization structure.  It allows
one process to wake up one or more processes that are waiting for some
condition to become true.
*/

class Condition {
  public:
    Condition(Mutex*);
    ~Condition();

    void wait();
    // requires - this thread has grabbed the mutex specified at creation
    void signal();
    void broadcast();
  private:
    Condition_Rep* rep;
};

#endif /* _THREAD_H */
