// Copyright 1995 Barbara Liskov

/*
\section{Transaction Manager}

The TM interface is used for transaction management.

\subsection{Todo}

\begin{itemize}
\item arguments to commit interface
\item What happens if fe connection dies while commit is in progress?
\item Destructor?
\item Everything ...
\end{itemize}
*/

#ifndef _TM_H
#define _TM_H

#include "common/xref.h"
#include "common/tid.h"
#include "common/or_index.h"
#include "common/modset.h"
#include "mm/logindex.h"

class FE_manager;
class MM;
class Mutex;
class Transaction;
class Validate_queue;
class Log;
class Rinfo;
class Xrefs;
class Uids;
class Tstamp_Generator;
class Mos;
class Nos;
class Installer;
class Prepared_Log_Record;
class Coord_set;
class OR_set;

struct or_stat;

class TM {
  public:
    TM();
    // effects - create a transaction manager from configuration information

    ~TM ();
    // Destructor

    ubits32 commit_single(FE_manager* fe, Transaction const* tx,
			  Xrefs*, Uids*);
    // Should raise an exception for an abort. Has to be done.
    // requires - Transaction involves only this OR.
    // returns  - The result of the commit
    //            Puts new object information in xrefs and uids
    // effects  - commit the single-OR transaction tx and return the outcome.  
    //            This call is allowed to return before the commit is 
    //            finished as long as the outcome message is delivered 
    //            to the fe eventually.
    //            If the fe connection dies while the commit is being
    //            done, the commit work still proceeds
    //            Adds appropriate records to log.

    ubits32 prepare_coord(FE_manager* fe, Transaction const* tx,
		    Xrefs*, Uids*);
    // requires - This OR is the coordinator for tx.
    // returns  - The result of the prepare:
    //            OR_COMMITTED if transaction has committed everywhere.
    //            OR_STALEABORT if transaction aborted.
    //            0 otherwise (prepare was successful).
    //            Puts new object information in xrefs and uids on success.
    // effects  - Performs validation of the multiple-OR transaction tx.
    //            Adds appropriate records to log.

    ubits32 prepare_part(FE_manager* fe, Transaction const* tx,
		    Xrefs*, Uids*, bool &force);
    // requires - This OR is not the coordinator for tx.
    // returns  - The result of the prepare:
    //            OR_COMMITTED if transaction prepared successfully.
    //            OR_STALEABORT if transaction aborted.
    //            Puts new object information in xrefs and uids on success.
    // effects  - Performs validation of the multiple-OR transaction tx.
    //            Adds appropriate records to log.
    //            Sets force: true iff transaction prepared but prepare
    //            record was forced to disk (for Stamos optimization).

    void abort(Tid const& tid);
    // effects  - Abort given transaction.  If we are coordinator, send 
    //            abort messages to participants who have voted for 
    //            transaction to commit.
    //            Remove entry for transaction from or->tstatus.

    void commit(Tid const& tid);
    // requires - This OR is coordinator for tid.
    //            All participants have voted OK for transaction.
    // effects  - Commit given multiple-site transaction.

    void truncate_vqueue_if_needed();
    // effects: Remove "old" entries from the VQ and update threshold.

    void stat(or_stat& s);
    // modifies	s
    // effects	Store the number of transactions committed so far in "s".

  private:
    Global_Tstamp  threshold;       // No trans is allowed below this ts.
    Global_Tstamp  stable_threshold; 
    // The stable copy of threshold.  Will not need to modify now.
    // Invariant : read_threshold <= stable_threshold.
    Global_Tstamp  to_install_stable_threshold;
    // This timestamp is the threshold value that has to be put on
    // stable storage. *After* this value has been flushed to the
    // log, the stable_threshold value is updated.
    // Invariant: stable_threshold <= to_install_stable_threshold

    Tstamp_Generator*   tgen;	    // Generator for local time stamps
    Mutex *mutex_validation;        // Mutex during the whole validation
				    // process. 
    Validate_queue*     vq;         // Validation Queue.
    OR_Index            single_or_index; // Used in single-OR transactions
    Coord_set*          coord_set;  // Set of coordinators used in Stamos'
				    // optimization.
    Tstamp              last_trunc; // Last time coordinator set was truncated

    ubits32             xactions;   // Number of transactions committed so far.

    friend Installer;   // Let update thread use internal methods
    
    // \subsection{Private methods}

    Rinfo* validate (FE_manager* fe, Transaction const* tx,  
		     Global_Tstamp& tx_tstamp, Xrefs*, Uids*);
    // requires    mutex_validation to have been grabbed
    // effects --- Validate the transaction and return true if it succeeds
    //             Puts the new xrefs and uids in "xrefs" and "uids".
    //             tx->mos and tx->nos have not been deleted.
    // returns --- null if the validation fails
    //             else the reservation information.

    bool threshold_check (Transaction const* tx, Global_Tstamp& tstamp);
    // requires    mutex_validation to have been grabbed
    // effects --- Checks if the transaction has a timestamp greater
    // than the threshold.

    bool invalid_obj_check (FE_manager* fe, Transaction const* tx); 
    // requires    mutex_validation to have been grabbed
    // effects --- Checks if the transaction has read any stale objects.
    //             using the FE tables of fe.

    Rinfo* reserve_object_space (Transaction const* tx, Xrefs*, Uids*);
    // requires    mutex_validation to have been grabbed
    // effects --- Reserve space for all the MOS objects.
    //             Puts the new xrefs and uids in "xrefs" and "uids"
    // returns --- Resv info if it succeeds, else returns null.

    bool update_threshold(Global_Tstamp &tstamp);
    // requires    mutex_validation to have been grabbed
    // effects     Sets threshold to maximum of current value and value of
    //             given timestamp.  If stable_threshold is updated, returns
    //             true; otherwise returns false.

    void set_stable_threshold(bool mutex_held);
    // requires  mutex_validation has been grabbed iff mutex_held is TRUE
    //           Called after the log has been flushed
    // effects   Sets the stable_threshold to the value that is on disk
    //           If mutex is not held, will grab it and then release it.
    //           Otherwise it will not relase the mutex

    void add_to_invalid_sets (FE_manager *fe, const Oref* wset, int wsize);
    // requires    mutex_validation to have been grabbed
    // modifies    or->fe_info_set
    // effects     Enters into the FE_tables of other FEs, all of
    //		   the objects in wset (wsize is the size of the array).

    void add_to_FE_table (FE_manager *fe, Xrefs *xrefs);
    // modifies    or->fe_info_set
    // effects     Enters the given objects into the FE table 
    //             of the given FE, if the given FE is connected.  
    //             This is used for new objects created by a transaction.

    void truncate_coordinators();
    // requires    Caller holds validation mutex.
    // effects     Removes old coordinators from coordinator set for
    //             Stamos' optimization.
    
    // \subsection{Logging operations}
    // These operations create log records, add them to the log, and
    // flush the log if necessary.

    void mark_installed(Global_Tstamp* ts);
    // requires: Transaction corresponding to ts exists
    //           hould not be called by an internal routine of TM
    // effects:  Marks the corresponding transaction as
    //           installed

    void log_commit_single(Transaction const* tx, Rinfo* r, Xrefs *xrefs,
			   FE_manager* fe);
    // requires    mutex_validation to have been grabbed
    // effects --- Log the transaction commit information for single OR trans.
    //		   and install any necessary changes.
    //             fe refers to the FE whose transaction is being committed
    //             Releases mutex_validation so that it is not held
    //             during logging.

    Log_Index log_prepared(Transaction const* tx, Rinfo *r, Xrefs *xrefs,
			   bool &force);
    // requires    mutex_validation to have been grabbed
    // effects --- Log the transaction prepare information at participant.
    //             Releases mutex_validation so that it is not held
    //             during logging.
    //             If force is true, record will be forced to disk.
    //             If force is false, force will be set to true if record
    //               is forced to disk.
    // returns --- Log index of prepare record.

    Log_Index log_committing(Transaction const* tx, Rinfo *r, Xrefs *xrefs);
    // requires    mutex_validation to have been grabbed
    // effects --- Log the transaction prepare information at coordinator.
    //             Releases mutex_validation so that it is not held
    //             during logging.
    // returns --- Log index of committing record.

public:
    void log_committed(Tid const&, Xrefs*, OR_Index*, OR_set* participants);
    // effects --- Log a committed record for given transaction.
    
    void log_abort(Tid const &tid);
    // effects --- Log an abort record for given transaction.

    void log_done(Tid const &tid);
    // effects --- Log a done record for given transaction.

    void log_participant(Tid const &tid, OR_num participant, 
			 Prepared_Log_Record *rec, Log_Index index);
    // effects --- Log participant's prepare record in coordinator log 
    //             protocol.
};

#endif /* _TM_H */
