# Inquir-specific code

send_user "\nInquir Tests Started\n"

proc spawn_inquir {or_id fe_id client_id fe_name commit_freq dummy_data} {
    # requires: or_id refers to the OR that the FE will talk to
    # effects: Spawn an inquir process and set the ID of the FE
    #          and the client in fe_id and client_id. Sets fe_name
    #          to be the fe_name outputted by the FE at startup
    #          commit_freq indicates how often transactions are committed
    #          i.e. after every operation, 2 operations etc
    #          If dummy_data is TRUE, inquire does not prompt for field data

    upvar $fe_id FE_ID
    upvar $client_id CLIENT_ID
    upvar $fe_name FE_NAME

    set data_option ""
    if {$dummy_data} {
	set data_option "-d"
    }
    spawn_client $or_id FE_ID CLIENT_ID FE_NAME inquirTest\
	"-D1 -m 1 -s 1700 -i" "-T $commit_freq $data_option"
}

proc send_user_info {values_list client_id} {
    # requires: values_list is a list with field values in the right order
    #           (as the inquir code requires them)
    # effects: Sends the values to the client

    set odd 0
    foreach val $values_list {
	if {$odd} {
	    send -i $client_id "$val\n"
	}
	set odd [expr !$odd]
    }
}

proc get_user_info {client_id username results_arrname} {
    # effects: Send a "show" command to client_id for username and put the
    #          results in  results. If no username exists results is not
    #          filled. The results fields are set as specified in the global
    #          "orig_values_array". XXX Need to state this better

    upvar $results_arrname results

    set command_found 0
    set results_found 0
    # Send show command to client
    send -i $client_id "s $username 1\n"
    set field_re "\n(\[a-zA-Z ()\]+):- "
    set value_re "(\[a-zA-Z0-9_/ \#()@:\.\-\]+)\r"
    while 1 {
	expect {
	    -i $client_id -re "device stream: Interrupted system call" {
		send_user "Error: Device stream error in get_user_info\n"
		exit
	    }
	    -i $client_id -re "Command: s, Name: $username, Num: 1" {
		set command_found 1
	    }
	    -i $client_id -re "$field_re$value_re" {
		set results($expect_out(1,string)) $expect_out(2,string)
		set results_found 1
		# send_user "<$expect_out(1,string)>-- <$expect_out(2,string)>\n"
	    }
	    -i $client_id -re "INPUT:" {
		if {$command_found} break;
	    }
	}
    }
    if {!$results_found} {
	th_assert 0 "User information was not found in get_user_info"
    }
# XXX Should check that the transaction committed or aborted
}


proc check_field_values {client_id username orig_values_array} {
    # requires: orig_values_array refers to a valid associative array that
    #           contains information about username (all fields must be valid)
    # effects: Sends a "show" command to the client (client_id) for user
    #          username. Waits for output from the client (client_id) and
    #          compares the field values outputted to the ones in
    #          orig_values_array.

    upvar $orig_values_array orig_values

    get_user_info $client_id $username results

    # Check that the field values were as expected 
    foreach fieldname [array names orig_values] {
	th_assert [expr ![string compare $results($fieldname)\
			      $orig_values($fieldname)]] "$fieldname mismatch"
    }
}

proc check_with_client {or_id fe_id client_id fe_name orig_values
    username read written new errormsg} {
    # effects: if client_id or fe_id is < 0, creates a new client and checks
    #          for the fields of user "username". The orig_values are in the
    #          array "orig_values". On the OR side it waits for
    #          a commit corresponding to this FE (checks the read, written and
    #          new values). Closes the connections to FE and client before
    #          returning 
    #
    #          If client_id and fe_id are>= 0, it does not create a new client
    #          but uses this client_id for checking the transaction
    #          It also uses fe_name to wait on the OR for this FE's transaction
    #          In this case, does not close the connection with the client
    #          XXX This procedure's specs need to be imprved

    upvar $orig_values orig_values_array

    set new_proc [expr $fe_id < 0 || $client_id < 0]
    if {$new_proc} {
	spawn_inquir $or_id fe_id client_id fe_name 1 0
    }

    check_field_values $client_id $username orig_values_array
    assert_commit $or_id $fe_id $client_id $fe_name\
	$read $written $new $errormsg
    if {$new_proc} {
 	close_process $client_id
	close_process $fe_id
    }
}


proc check_creates_updates {or_id ifcreate readobjs writeobjs newobjs} {
    # requires: or_id corresponds to a newly created inquir db
    # effects: Checks creation/modification of a large number of
    #          objects. ifcreate indicates whether we are checking
    #          creation or modification of objects. ..objs indicate
    #          the number of objects read, written, new for the 1st transaction
    
    set result 1
    # Create an inquir client that will use dummy data for all the change and
    # insert operations, i.e. we do not have to send the data interactively
    spawn_inquir $or_id fe_id client_id fe_name 1 1
    if {$ifcreate} {
	set command "a"
	set comm "Creation"
	set wincr 1
	set nincr -2
    } else {
	set command "c"
	set comm "Modification"
	set wincr 0
	set nincr -1
    }

    set transnum 50
    send -i $client_id "$command new_name $transnum\n"
    # Check whether the OR got the right number of objects

    for {set count 0} {$count < $transnum} {incr count} {
	set res [assert_commit $or_id $fe_id $client_id $fe_name -1 -1 -1\
		     "Dummy message"]
	set actual_read [lindex $res 0]
	set actual_written [lindex $res 1]
	set actual_new [lindex $res 2]

	expect {
	    -i $client_id -re "Transaction" {}
	    -i $client_id -re "device stream: Interrupted system call" {
		set result 0
		break
	    }
	}
	if {$ifcreate && $actual_written != $writeobjs} {
	    # A new vector was created for an array so 1 less written object
	    # and 1 more new object
	    set w [expr $writeobjs - 1]
	    set n [expr $newobjs + 1]
	} else {
	    set w $writeobjs
	    set n $newobjs
	}
	th_assert [expr $actual_read == $readobjs]\
	    "Read incorrect (actual = $actual_read, predict = $readobjs)"
	th_assert [expr $actual_written == $w]\
	    "Write incorrect (actual = $actual_written, predict = $w)"
	th_assert [expr $actual_new == $n]\
	    "New incorrect (actual = $actual_new, predict = $n)"
	if {$count == 0} {
	    incr readobjs -2
	    incr writeobjs $wincr
	    incr newobjs  $nincr
	}
    }
    close_process $client_id
    close_process $fe_id
    
    if {$result} {
	send_user "Test: $comm of objects in multiple transactions ... ok\n" 
    } else {
	send_user "Warning:: Device stream error encountered\n"
    }
}

proc create_inquir_db {dbname} {
    # effects: Create a new inquire database with filename dbname

    set or_id [spawn_or $dbname 1 0 ""]
    spawn_inquir $or_id fe_id client_id fe_name 1 0
    
    set data(0) {
	"Phil Bogle" "pbogle" "pbogle" "February 6" "12069"
	"Phil" "pbogle@thor.lcs.mit.edu" "Programming Methodology Group"
	"Professor Barbara Liskov" "Graduate student" 
	"Thor" "NE43-526" "253-6212"
	"185 Freeman Street, #152, Brookline, MA  02146" "(617)566-3267"
	"pbogle@thor.lcs.mit.edu 21-Jul-1993 11:09:58" "pbogle" "N/A"
    }
    set data(1) {
	"Deborah Hwang" "debbieh" "debbieh" "April 8" "11139"
	"Debbie" "debbieh@kaon.lcs.mit.edu" "Programming Methodology Group"
	"Barbara Liskov" "Graduate Student"
	"Thor" "NE43-525" "253-3058"
	"303 Lowell St. #5; Somerville, MA 02145" "(617) 623-3563"
	"debbieh@kaon 10-Mar-1993 16:18:56" "DEBBIEH" "N/A"
    }
    set data(2) {
	"Robert Gruber" "gruber" "gruber" "September 11, 1965" "11118"
	"Bob" "gruber@thor.lcs.mit.edu" "Programming Methodology Group"
	"Liskov" "Graduate Student" 
	"THOR" "NE43-525" "253-3058"
	"Burton House, room 321" "225-8277"
	"gruber@thor.lcs.mit.edu 22-Jul-1992 13:46:44" "GRUBER, DCURTIS, PRJ" "N/A"}
    set data(3) {"Quinton Zondervan" "qyz" "qyz" "September 15" "11960"
	"Q" "qyz@harmony.lcs.mit.edu" "Programming Methodology Group"
	"Liskov" "Graduate student"
	"The Mighty Thor" "NE43 526" "253-6212"
	"11 Silvey Place, Somerville, MA 02143"	"623-0386"
	"qyz@harmony.lcs.mit.edu 21-Jan-1994 09:47:29" "QYZ" "N/A"
    }
    set data(4) {
	"Liuba Shrira" "liuba" "liuba" "1 Jan" "11141"
	"Liuba" "liuba@vilnius.lcs.mit.edu" "Programming Methodology Group"
	"Barbara Liskov" "Research Staff"
	"Distributed Systems" "NE43-537" "6101"
	"474 Broadway, Cambridge, MA 02138" "(617) 868 3651"
	"dcurtis@miki.lcs.mit.edu 10-Jul-1992 13:35:48"	"DCURTIS PRJ" "N/A"
    }
    set data(5) {
	"Atul Adya" "adya" "adya" "September 30" "11860"
	"N/A" "adya@harmony.lcs.mit.edu" "Programming Methodology Group"
	"Barbara Liskov." "Graduate Student"
	"Thor" "NE43-526 (Lab for Computer Science)" "(617) 253-6212"
	"133 Webster Avenue, #1, Cambridge, MA 02141" "(617)-864-5041"
	"adya@thor.lcs.mit.edu 19-Jul-1993 16:01:50" "ADYA" "N/A"
    }
    set data(6) {
	"Umesh Maheshwari" "umesh" "umesh" "N/A" "11764"
	"N/A" "umesh@tonic.lcs.mit.edu"	"Programming Methodology Group"
	"Barbara Liskov" "Graduate student"
	"Thor" "NE43-527"  "(617) 253-6015"
	"341 Hurley St #3, Cambridge MA 02141" "(617) 492-3717"
	"umesh@note.lcs.mit.edu 1-Nov-1993 16:55:26" "UMESH" "N/A"
    }
    set data(7) {
	"Mark Day" "mday" "mday" "N/A" "11042"
	"Mark-sub-D" "mday@clef.lcs.mit.edu" "Programming Methodology Group"
	"fun and profit" "Graduate Student"
	"systems" "NE43-527" "(617) 253-6015"
	"27 Forest Street #3, Cambridge, MA 02140" "(617) 497-5325"
	"mday@note.lcs.mit.edu 28-Jan-1994 17:19:27"
	"MDAY" "The universe is not only stranger than we imagine, it is stranger than we can imagine."
    }
    set data(8) {
	"Paul R. Johnson" "prj" "prj" "N/A" "11007"
	"N/A" "prj@gamba.lcs.mit.edu" "Programming Methodology Group"
	"Prof. Liskov" "Research staff"
	"Harp and Thor"	"NE43-531" "3-1945"
	"47 Hillside Rd., Watertown" "923-9068"
	"prj@gamba.lcs.mit.edu 22-Jul-1992 13:22:36" "PRJ DCURTIS" "N/A"
    }
    set data(9) {
	"Helen Chen" "hhchen" "hhchen" "N/A" "12393"
	"N/A" "hhchen@athena.mit.edu" "Programming Methodology Group"
	"Curtis" "Undergrad"
	"N/A" "N/A" "N/A" "N/A"	"N/A"
	"dcurtis@miki.lcs.mit.edu 2-May-1994 14:21:39" "N/A" "N/A"
    }
    set data(10) {
	"Barbara Liskov" "liskov" "liskov" "N/A" "11039"
	"N/A" "liskov@thor.lcs.mit.edu"	"Programming Methodology Group"
	"N/A" "Faculty"
	"N/A" "NE43-528" "253-5886"
	"N/A" "N/A"
	"dcurtis@miki.lcs.mit.edu 4-Mar-1994 10:42:29" "DCURTIS PRJ" "N/A"
    }
    set data(11) {
	"Mike Malone" "mmalone" "mmalone" "N/A"	"12389"
	"N/A" "mmalone@miki.lcs.mit.edu" "Programming Methodology Group"
	"Liskov" "Support staff"
	"N/A" "NE43-529" "253-5886"
	"N/A" "N/A"
	"dcurtis@miki.lcs.mit.edu 13-Apr-1994 15:11:30"	"N/A" "N/A"
    }
    set data(12) {
	"Andrew Myers" "andru" "andru" "N/A" "11754"
	"N/A" "andru@tonic.lcs.mit.edu"	"Programming Methodology Group"
	"Barbara Liskov" "Graduate Student"
	"N/A" "NE43-527" "6015"
	"N/A" "N/A"
	"andru@thor.lcs.mit.edu 15-Feb-1994 15:30:33" "ANDRU DCURTIS" "N/A"
    }
    set data(13) {
	"Sanjay Ghemawat" "sanjay" "sanjay" "October 28" "11152"
	"N/A" "sanjay@clef.lcs.mit.edu"	"Programming Methodology Group"
	"Barbara Liskov" "Graduate Student"
	"Thor" "NE43-525" "3058"
	"20 Hubbard Avenue #3, Cambridge MA 02140" "491-7616"
	"sanjay@jukebox.lcs.mit.edu 1-Sep-1993 11:54:13" "SANJAY" "N/A"
    }
    set data(14) {
	"Dorothy W. Curtis" "dcurtis" "dcurtis"	"N/A" "11008"
	" " "dcurtis@miki.lcs.mit.edu" "Programming Methodology Group"
	"Liskov" "Research Staff"
	"N/A" "NE43-531" "253-0541"
	"N/A" "N/A"
	"dcurtis@miki.lcs.mit.edu 12-Jan-1993 10:40:05"	"DCURTIS" "N/A"
    }
    # Set up as -- 0 1 2 3 4     5 6 7     8 9   10 11 12    13 14    15 16 17

    # Set up a list for the order in which the fields are sent
    set send_order {0 4 5 3 6 11 12 10 8 7 9 13 14 16 17}

    set num_people 15
    for {set i 0} {$i < $num_people} {incr i} {
	# Send the data for this person
	set name [lindex $data($i) 1]
	expect -i $client_id -re "INPUT:"
	send -i $client_id "a $name 1\n"
	foreach num $send_order {
	    send -i $client_id "[lindex $data($i) $num]\n"
	}
	expect -i $or_id -re "Transaction .* Committed"
    }
    set final_name [lindex $data([expr $num_people - 1]) 1]
    send -i $client_id "s $final_name 1\n"
    expect -i $or_id -re "Transaction .* Committed"
    expect -i $client_id -re "INPUT:"
    if {[regexp $expect_out(buffer) "User not found"]} {
	send_user "Database creation failed: Last user not found\n"
	exit
    }

    send_user "Database creation ... ok\n"

    close_process $fe_id
    close_process $client_id
    close_process $or_id

}
# ---------------------------------------------------------------------------
#                              TESTS START HERE 
# ---------------------------------------------------------------------------

# Create a database first
create_inquir_db inquir_th

# Start OR and a client
set OR_ID [spawn_or inquir_th 0 0 ""]
spawn_inquir $OR_ID FE_ID1 CLIENT_ID1 FE_NAME1 1 0

# Set user information to be used for sending to client
set orig_values_list {
    "Name"                        "Name1"
    "User Id"                     "Id2"
    "Nickname"                    "Nickname3"
    "Birthday"		          "Birthday4"
    "Email Address"	          "Email5"
    "Work Address"	          "Work_Address6"
    "Work Phone"	          "Work_Phone7"
    "Project"		          "Project8"
    "Supervisor"	          "Supervisor9"
    "Group"		          "Group10"
    "Relation" 		          "Relation11"
    "Home Address"	          "Home_Address11"
    "Home Phone"	          "Home_Phone12"
    "Modifiable by (Kerb)"        "Modifiable14"
    "Remarks"		          "Remarks15"
}

array set orig_values_array $orig_values_list
# Check if the handles are being freed correctly
send -i $CLIENT_ID1 "F now 1\n"
assert_commit $OR_ID $FE_ID1 $CLIENT_ID1 $FE_NAME1\
    5 0 0 "Free handle command (now)"
check_free_handles $FE_ID1 $CLIENT_ID1 10
# 10 handles is due to -- 1 invalid + 1 special var + 5 variable declarations +
# static variables (1 each) in th_directory::Class in th_inquirbase::Class
# th_interface::Class Note: the variable usr contains a reference to invalid

# ---- CHECK NEW OBJECT CREATION ----
# Create new user called new1. Wait on the OR for the commit to succeed
# Also check for freeing of handles
send -i $CLIENT_ID1 "F write 1\n"
assert_commit $OR_ID $FE_ID1 $CLIENT_ID1 $FE_NAME1\
    3 0 0 "Free handle command (write)"
send -i $CLIENT_ID1 "a new1 1\n"
send_user_info $orig_values_list $CLIENT_ID1
check_free_handles $FE_ID1 $CLIENT_ID1 12
# 2 more handles added due to a static variable in set_user and a normal one
assert_commit $OR_ID $FE_ID1 $CLIENT_ID1 $FE_NAME1\
    1 2 20 "Addition of new user failed"

# Check if these objects were actually created using another client
check_with_client $OR_ID -1 -1 0 orig_values_array new1 6 0 0 "New objects"

send_user "Test: New object creation ... ok\n"
send_user "Test: Freeing of handles by client (1) ... ok\n"

# ---- CHECK OBJECT READ/WRITES ----

# Change attibutes of user "adya". Wait on OR for commit to succeed
send -i $CLIENT_ID1 "c adya 1\n"
send_user_info $orig_values_list $CLIENT_ID1
assert_commit $OR_ID $FE_ID1 $CLIENT_ID1 $FE_NAME1\
       3 1 15 "Update of existing user"
check_with_client $OR_ID $FE_ID1 $CLIENT_ID1 $FE_NAME1\
    orig_values_array adya 4 0 0 "Read objects"

# Check if freeing of handles is working correctly after doinf a read 
send -i $CLIENT_ID1 "F read 1\n"
assert_commit $OR_ID $FE_ID1 $CLIENT_ID1 $FE_NAME1\
    3 0 0 "Free handle command (read)"
send -i $CLIENT_ID1 "s adya 1\n"
assert_commit $OR_ID $FE_ID1 $CLIENT_ID1 $FE_NAME1\
    4 0 0 "Dummy read"
check_free_handles $FE_ID1 $CLIENT_ID1 11

send_user "Test: Freeing of handles by client (2) ... ok\n"

close_process $CLIENT_ID1
close_process $FE_ID1
# Check if objects were actually created using another client
check_with_client $OR_ID -1 -1 0 orig_values_array adya 6 0 0 "Mod objects"

send_user "Test: Reads/modifications of existing objects ... ok\n"


# Check if transaction aborts work along with reverting of modified objects

spawn_inquir $OR_ID FE_ID1 CLIENT_ID1 FE_NAME1 3 0
get_user_info $CLIENT_ID1 sanjay results
send -i $CLIENT_ID1 "c sanjay 1\n"
send_user_info $orig_values_list $CLIENT_ID1
# Abort transaction and check that object reverted to original status
send -i $CLIENT_ID1 "A \n"
check_field_values $CLIENT_ID1 sanjay results
close_process $CLIENT_ID1
close_process $FE_ID1
send_user "Test: Transaction aborts and reverting to old values ... ok\n"


# CHECK DATA WITH NEW OR
close_process $OR_ID
set OR_ID [spawn_or inquir_th 0 0 ""]
check_with_client $OR_ID -1 -1 0 orig_values_array new1 6 0 0 "New OR new1"
check_with_client $OR_ID -1 -1 0 orig_values_array adya 6 0 0 "New OR adya"
send_user "Test: Whether object updates become persistent ... ok\n"
close_process $OR_ID


# ----------------------- MULTI-CLIENT TESTS --------------------------------

# Start new OR and 2 clients
set OR_ID [spawn_or inquir_th 0 0 ""]
spawn_inquir $OR_ID FE_ID1 CLIENT_ID1 FE_NAME1 1 0
spawn_inquir $OR_ID FE_ID2 CLIENT_ID2 FE_NAME2 1 0

# Check if invalidations are being sent and received (acks also)
# Perform modifications on the same object. Let client 1 to succeed. Client2's
# transaction aborts and we check on this client's connection for abort etc

send -i $CLIENT_ID1 "c adya 1\n"
send -i $CLIENT_ID2 "c adya 1\n"
# Send a command to client 2 and wait till it replies (to ensure that
# the transaction has started)
send -i $CLIENT_ID2 "New name\n"
expect -i $CLIENT_ID2 -re "User Id:"

# Send the new values to the frst client
send_user_info $orig_values_list $CLIENT_ID1
set abort_exc 0
# No. of fields that exist for a user
set no_fields [expr [llength $orig_values_list] / 2]
set fieldno 1
set timeout 3
# Wait for abort messages to be printed out by FE 2 and client 2
# Break out of the loop if have sent all fields to client 2
# OR when client 2 actually succeeds in sending a commit request to FE
# (and received a reply)
while 1 {
    expect {
	-i $FE_ID2 -re "Immediate abort at FE" {
	    set abort_exc 1
	    }
	-i $CLIENT_ID2 -re "Abort exception received" {
	    set abort_exc 1
	}
	-i $CLIENT_ID2 -re "Aborted = 0" {
	    # Transaction did not abort
	    break
	}
	-i $CLIENT_ID2 -re "Transaction No:" {
	    break;
	}
	timeout {
	    # Keep sending field values to the client till it aborts
	    # If current transaction does not abort, just quit
	    if {$fieldno  == $no_fields} break;
	    incr fieldno
	    send -i $CLIENT_ID2 "New field\n"

	}
    }
}
set timeout -1

# Check if the OR sent and received the invalidation messages
set sent_invalid 0
set ack_invalid 0
set timeout_count 0
set timeout 1
while 1 {
    expect {
	-i $OR_ID -re "Sent Inv message: S: 1 E:1 Num: 1" {
	    set sent_invalid 1
	}
	-i $OR_ID -re "Received Inv Ack. Msg seen: 1" {
	    set ack_invalid 1
	    break;
	}
	timeout {
	    incr timeout_count
	    if {$timeout_count > 6} break
	    # Ack was not received for a minute or so
	}
    }
}
set timeout -1

th_assert $sent_invalid "Invalidation message not sent by OR"
send_user "Test: Sending of invalidation messages ... ok\n"
if {!$ack_invalid} {
    send_user "NOTE: Acknowledgement of invalidation messages not tested\n"
} else {
    send_user "Test: Acknowledgement of invalidation messages ... ok\n"

}

if {!$abort_exc} {
    send_user "NOTE: Abort exceptions not tested\n"
} else {
    send_user "Test: Abort exceptions ... ok\n"

}

# Check if the user got modified by the first client without problems 
check_with_client $OR_ID -1 -1 0 orig_values_array adya 6 0 0 "Multi-client 1"
# The following message needs to be fixed XXX
send_user "Test: Modifications in presence of multiple clients and\
 aborts ... ok\n"

# Check if client 2 is still functioning
if {$abort_exc} {
    # Note that this also checks whether reverting at a frontend is working or
    # not because we had modified the name field at CLIENT 2
    # XXX Does not check reverting since the object concerned has been
    # invalidated

    check_with_client $OR_ID $FE_ID2 $CLIENT_ID2 $FE_NAME2 orig_values_array\
	adya 4 0 0 "Client 2 not functioning correctly after abort"
    send_user "Test: Client functioning after abort ... ok\n"
}

close_process $CLIENT_ID1
close_process $CLIENT_ID2
close_process $FE_ID1
close_process $FE_ID2
close_process $OR_ID

# Run 2 clients for 200 or 300 transactions
set OR_ID [spawn_or inquir_th 0 0 ""]
# Client 1 will keep running infinitely whereas for client 2
# we will send the data. The data is "c adya 1", so that
# we do not have to worry whether the program expects field data or command
spawn_inquir $OR_ID FE_ID1 CLIENT_ID1 FE_NAME1 1 1
spawn_inquir $OR_ID FE_ID2 CLIENT_ID2 FE_NAME2 1 0
expect -i $CLIENT_ID1 "INPUT:"
send -i $CLIENT_ID1 "c adya -1\n"

set comcount 0
set inv_msgs 0
set commits 0
set device_stream_error 0
set result [catch {
    while 1 {
	expect {
	    -i $CLIENT_ID2 -re "Received Inv message:" {
		incr inv_msgs
	    }
	    -i $CLIENT_ID2 -re "default|INPUT:|Transaction No:" {
		send -i $CLIENT_ID2 "c adya 1\n"
		incr comcount
	    }
	    -i $CLIENT_ID2 -re "device stream: Interrupted system call" {
		set device_stream_error 1
		send_user "Error: Device stream error on the slower client\n"
		exit
	    }
	    -i $CLIENT_ID1 -re "Committed" {
		incr commits
	    }
	    -i $OR_ID -re "Transaction" {}
	}
	if {$comcount > 100} break;
    }
} errorstring]

if {$result} {
    th_assert 0 "FAILED: Client, FE or OR died"
}

if {$commits && $inv_msgs} {
    send_user "Test: Number of updates by mutliple clients on\
    the same object ... ok\n"
} else {
    send_user "Note: Could not test large number of  modifications by\
    mutliple clients on the same object, Commits = $commits, Inv = $inv_msgs\n"
}

close_process $CLIENT_ID1
close_process $CLIENT_ID2
close_process $FE_ID1
close_process $FE_ID2
close_process $OR_ID

# ------------------------------------------------------------------

# Check whether adding a lot of objects and changing them works ok
create_inquir_db inquir_th
set OR_ID [spawn_or inquir_th 0 0 ""]
#log_user 1
check_creates_updates $OR_ID 1 3 2 6
check_creates_updates $OR_ID 0 5 1 1
close_process $OR_ID
#log_user 0
