# This test tries to stress the OR. It forces the OR to
# use a very small log. It does the following:
# Create lots of new objects to cause disk writes:
#  - Check values of objects that should have been flushed out of the log.
#  - Then crash, recover, and check values.
# Check that objects are accessible even while being written out:
#  - start modifying set of objects in round-robin fashion
#  - check object values often

send_user "\nOR Stress Tests Started\n"
send_user "This test takes about 10 minutes to run\n"

proc spawn_inquir {or_id fe_id client_id fe_name commit_freq dummy_data} {
    # requires: or_id refers to the OR that the FE will talk to
    # effects: Spawn an inquir process and set the ID of the FE
    #          and the client in fe_id and client_id. Sets fe_name
    #          to be the fe_name outputted by the FE at startup
    #          commit_freq indicates how often transactions are committed
    #          i.e. after every operation, 2 operations etc
    #          If dummy_data is TRUE, inquire does not prompt for field data

    upvar $fe_id FE_ID
    upvar $client_id CLIENT_ID
    upvar $fe_name FE_NAME

    set data_option ""
    if {$dummy_data} {
	set data_option "-d"
    }
    spawn_client $or_id FE_ID CLIENT_ID FE_NAME inquirTest\
	"-D1 -m 1 -s 2000 -i" "-T $commit_freq $data_option"
}

proc send_items {client_id values_list} {
    # requires: values_list is a list with field values in the right order
    #           (as the inquir code requires them)
    # effects: Sends the values to the client

    foreach val $values_list {
        send -i $client_id "$val\n"
    }
}

proc get_items {or_id client_id username} {
    # effects: Send a "show" command to client_id for username and put the
    #          results in  results. If no username exists results is not
    #          filled. The results fields are set as specified in the global
    #          "orig_values_array". XXX Need to state this better

    #empty list
    set results {}

    set command_found 0
    set results_found 0

    set field_re "\n(\[a-zA-Z ()\]+):- "
    set value_re "(\[a-zA-Z0-9_/ \#()@:\.\-\]+)\r"

    expect -i $client_id -re "Command: s, Name: $username"
    set i 0
    while {$i < 15} {
        expect -i $client_id -re "$field_re$value_re" {
            lappend results $expect_out(2,string)
            set results_found 1
            #send_user "<$expect_out(1,string)>-- <$expect_out(2,string)>\n"
        }
        incr i
    }
        
    expect -i $client_id -re "Transaction No:"
    expect -i $or_id -re "Transaction"

    if {!$results_found} {
	th_assert 0 "User information was not found in get_user_info"
    }
    # XXX Should check that the transaction committed or aborted
    return $results
}

proc check_user_values {orig_list comp_list} {

    set index 0
    foreach field $orig_list {
        th_assert [expr ![string compare $field [lindex $comp_list $index]]] \
            "$field mismatch"
        incr index
    }
}

proc construct_name {i} {
    if {$i == 0} {
        return "name"
    } else {
        return "name$i"
    }
}  


proc construct_value_list {i} {
    return [list          \
                "name$i"          \
                "id$i"            \
                "nickname$i"      \
                "birthday$i"      \
                "email$i"         \
                "workaddress$i"   \
                "workphone$i"     \
                "project$i"       \
                "supervisor$i"    \
                "group$i"         \
                "relation$i"      \
                "homeaddress$i"   \
                "homephone$i"     \
                "modifiable$i"    \
                "remarks$i"       \
               ]
}

# ---------------------------------------------------------------------------
#                              TESTS START HERE 
# ---------------------------------------------------------------------------

#OR Options.Set log to be very small (60K)
set or_options "-l 60"

# ---- BEGIN Test Creation of a lot of new objects ----

# Start OR and a client
set OR_ID [spawn_or inquir_th 1 0 $or_options]
spawn_inquir $OR_ID FE_ID CLIENT_ID FE_NAME 1 0

# Create a lot of new objects

set NUM_NEW 500

send_user "Test: Inserting $NUM_NEW people into the database ..."

send -i $CLIENT_ID "a name $NUM_NEW\n"

set i 0
while {$i < $NUM_NEW} {

    expect -i $CLIENT_ID "Name"
    expect -i $CLIENT_ID "Name"
    send_items $CLIENT_ID [construct_value_list $i]

    expect -i $OR_ID  -re "Transaction"
    expect -i $OR_ID  -re "Committed"

    incr i
}

expect -i $CLIENT_ID "INPUT"

send_user " ok\n"

send_user "Killing OR, client and restarting them\n"

# kill OR & client
close_process $CLIENT_ID
close_process $FE_ID
close_process $OR_ID


# Start OR (do not create new database) and a client
set OR_ID [spawn_or inquir_th 0 1 $or_options]
send_user "Test: Get data of $NUM_NEW people to see if all fields match ..."
spawn_inquir $OR_ID FE_ID CLIENT_ID FE_NAME 1 0

send -i $CLIENT_ID "s name $NUM_NEW\n"

set i 0
while {$i < $NUM_NEW} {
    check_user_values [construct_value_list $i] \
        [get_items $OR_ID $CLIENT_ID [construct_name $i]]
    incr i
}

send_user " ok\n"

# ---- END Test Creation of a lot of new objects ----

# ---- BEGIN Test Modification of objects in round-robin fashion ----

send_user "Test: Modify data of $NUM_NEW people and see if all fields match ..."

set name_i 0
set read_i 0
set write_i $NUM_NEW

set NUM_ROUND 2

while {$read_i < ($NUM_NEW * $NUM_ROUND)} {

    send -i $CLIENT_ID "s [construct_name $name_i] 1\n"
    check_user_values [construct_value_list $read_i] \
        [get_items $OR_ID $CLIENT_ID [construct_name $name_i]]

    expect -i $CLIENT_ID "INPUT"

    send -i $CLIENT_ID "c [construct_name $name_i] 1\n"
    expect -i $CLIENT_ID "Name"
    expect -i $CLIENT_ID "Name"
    send_items $CLIENT_ID [construct_value_list $write_i]
    
    global num_regexp
    expect -i $CLIENT_ID -re "Transaction No. $num_regexp" {
	set tnum $expect_out(1,string)
    }
    expect -i $CLIENT_ID "INPUT"
    expect -i $OR_ID "Transaction"
    expect -i $OR_ID -re "Committed"

    incr read_i
    incr write_i
    incr name_i
    if {$name_i == $NUM_NEW} {
        set name_i 0
    }
}

send_user " ok\n"


# ---- END Test Modification of objects in round-robin fashion ----

close_process $CLIENT_ID
close_process $FE_ID
close_process $OR_ID
