#!/usr/local/bin/expect --

proc usage {{reason {}}} {
    if [string compare $reason {}] {
	send_error -- "thortest.tcl: $reason\n\n"
    }

    send_error {Usage: thortest.tcl [-v] [-V] [-c oo7] [-p port] [-d dir]
	-v                : Verbose output (default off)
	-V                : Extremely verbose (default off)
	-c <oo7>          : Whether the new db for OO7 should be created
			    (Default is not to create unless necessary)
	-p <port>         : Server port number (default = 0)
	-d <directory>    : Top level directory for thor executables
			    (default TOP = $HOME/thor/alpha)
			    Database is created/searched in TOP/or
    }
    send_error "\n"
    exit 1
}

proc debug_or {or_id} {
    # effects: start up an xterm and attach the debugger to the OR running
    #          with spawn id "or_id"
    global TOP
    set real_id [exp_pid -i $or_id]
    eval [list spawn xterm -e decladebug $TOP/or/or -pid $real_id]
    set spawn_id $or_id
    send_user "press X after debugger is initialized"
    interact "X" return
}

proc spawn_or {name ifnew_db ts or_arg} {
    # effects: Start a new or with the name $name. A new database is created if
    #          if_newdb is set to true. If ts is 1 then the -c option is used 
    #          to create the database otherwise it uses the -i option. It 
    #          uses the port number specified by the user. If it fails, it 
    #          keeps trying with higher port numbers
    #          or_arg contains any additional arguments you want to send
    #          to the OR. Use "" if no additional arguments

    global TOP OR_SOURCE portnum env hostname

    # Even if user does not ask for a new database, try to create it if
    # none exists.
    if ![file exists $TOP/or/$name.db] {
	set ifnew_db 1
    }

    set pn $portnum
    set env(THOR) "$hostname:$pn"

    set trials 0

    # no additional arguments, have to set it to something
    if {![string compare $or_arg ""]} {
        set or_arg "-D 1"
    }
    while 1 {
	incr trials

	if {$ifnew_db} {
	    set curdir [pwd]
	    cd $OR_SOURCE
            if {$ts} {
                eval [list spawn or -p $pn -D 1 -c $OR_SOURCE/init/$name.or \
                          $or_arg $TOP/or/$name.db]
	    } else {
                eval [list spawn or -i -g 57 -p $pn -D 1 $or_arg  \
                          $TOP/or/$name.db]
	    }
	    cd $curdir
	} else {
	    eval [list spawn or -p $pn -D 1 $or_arg $TOP/or/$name.db]
	}
	set done 0

	# Wait for the server to start up or start a new one if this dies
	set result [catch {
	    expect "Server ready" {set done 1}
	} errorstring]

	if {$done} break;

	if {$trials >= 6} {
	    send_error "could not start OR: tried $trials times\n"
	    exit 1
	}

	incr pn
	set env(THOR) "$hostname:$pn"
    }
    if {$pn != $portnum} {
	send_user "Note: Port number $pn chosen\n"
    }
    return $spawn_id
}


proc spawn_client {or_id fe_id client_id fe_name client_name fe_args c_args} {
    # requires: client_name refers to a valid client executable
    #           fe_args are the arguments passed to the FE
    #           c_args are the arguments to be passed to the client. They
    #           should not specify the FE port/shared memory region explicitly
    #           or_id refers to the OR this FE will talk to
    # effects: Spawn a client process as specified by client_str. Put the
    #          spawn_id of the client and the FE in fe_id and client_id
    #          Sets fe_name to be the fe_name nformation printed out by FE
    
    upvar $fe_id FE_ID
    upvar $client_id CLIENT_ID
    upvar $fe_name FE_NAME
    global till_eol FE_NAME_REGEXP env

    # XXX Currently, not setting up a separate fe since parameters tofe in shm
    # and socket code are different. But we should oppen a separate FE so that
    # expect can destroy it if the client/OR die early

    set env(FE_FLAGS) "$fe_args"
    eval [list spawn $client_name] $c_args
    set CLIENT_ID $spawn_id
    set FE_ID $CLIENT_ID
    set timeout -1
    expect -i $FE_ID -re $FE_NAME_REGEXP {
	set FE_NAME $expect_out(1,string)
    }
    expect -i $or_id $FE_NAME
}

proc thor_cleanup {} {
    # effects: Called whenever the test file exits
    # puts stderr "Exited from test file\n"
}

proc get_objects_regexp {str} {
    # requires: str to be "Read", "Write" or "New"
    # effects: Get a regular expression for "str" that corresonds to the
    #          line outputted by the OR on receiving a commit
    return "\ *$str:\ *(\[0-9\]*).*\n"
}

proc assert_commit {orid feid clientid fe_name read written new str} {
    # requires: orid, feid and clientid are valid spawn_ids corresponding to
    #           the  OR, FE and client. fe_name is the fe_name
    #           that is outputted by the FE at startup time
    # effects: Waits for commit from $orid, gets the number of read, written
    #          and new objects and compares their values with "read",
    #          "written" and "new". Returns silenty if ok, otherwise,
    #          prints str on the screen and exits
    #          Keeps checking that the client, fe and or have not died
    #          If either of read, written or new is negative, it simply waits
    #          for the commit and returns
    # returns  a list of number of objects actually read, written and created

    global FE_NAME_REGEXP

    set objs(read) 0
    set objs(written) 0
    set objs(new) 0

    set rexpr [get_objects_regexp Read]
    set wexpr [get_objects_regexp Write]
    set nexpr [get_objects_regexp New]
    set tobreak 0
    set relevant_commit_received 0
    while 1 {
	set result [catch {
	    # In case the OR dies in the middle
	    expect {
		-i $orid -re $FE_NAME_REGEXP {
		    if {![string compare $fe_name $expect_out(1,string)]} {
			set relevant_commit_received 1
		    }
		}
		-i $orid -re "$rexpr$wexpr$nexpr" {
		    if {! $relevant_commit_received} {
			# send_user "IGNORED: $expect_out(0,string)\n"
			break
		    }
		    # send_user "FOUND: $expect_out(0,string)\n"

		    set objs(read) $expect_out(1,string)
		    set objs(written) $expect_out(2,string)
		    set objs(new) $expect_out(3,string)
		    set tobreak 1
		    break
		}
		timeout {
		    # Check if all processes are still alive
		    # send_user "TIMEOUT: $expect_out(0,string)\n"
		    th_assert [is_valid_spawnid $feid] "FE/Client died"
		    th_assert [is_valid_spawnid $clientid] "Client died"
		    th_assert [is_valid_spawnid $or_id] "Client died"
		
		}
	    }
	} ]
	if {$result} {
	    th_assert [is_valid_spawnid $orid] "OR died"
	}
	if {$tobreak} break
    }
    assert_objsnum objs $read $written $new $str
    return [list $objs(read) $objs(written) $objs(new)]
}

proc th_assert {cond str} {
    # effects: if cond is true, returns "OK\n" else sends str to the user
    #          and terminates the program

    if {$cond} {
	return "OK\n"
    } else {
	send_user "FAILED: $str\n"
	exit 1
    }
}

proc assert_objsnum {objsarr read written new str} {
    # effects: Ensures that number of objects read, written and new
    #          in objsarr is the same as given in the other parameters
    #          If this is not so, it exits the program after printing str
    #          If either of read, written or new is < 0, it simply  returns

    upvar $objsarr objs
    # send_user "$objs(read)  $objs(written) $objs(new)\n"
    if {$read < 0 || $written < 0 || $new < 0} { return }
    th_assert [expr $objs(read) == $read && $objs(written) == $written &&\
	       $objs(new) == $new] "Commit: $str"
}

proc check_free_handles {fe_id client_id num} {
    # requires: A free handle command has been sent to the client
    # effects: Checks that the number of handles that exist in the veneer
    #          is equal to num. (The various ids are the usual spwanids)
    #          If num is < 0, this procedure just waits for the output
    #          and does not compare the numbers
    
    global num_regexp
    expect {
	-i $client_id -re "Actual Entries =  $num_regexp"	{
	    if {$num >=0} {
		th_assert [expr $num == $expect_out(1,string)]\
		    "Wrong number of handles"
	    }
	}
	-i $client_id -re "device stream: Interrupted system call" {
	    set device_stream_error 1
	    send_user "Error: Device stream error in check_free_handles\n"
	    exit 1
	}
    }
}

proc is_valid_spawnid {id} {
    # effects: returns 1 if id is a valid spawnid else returns 0

    set timeout 0
    set result [catch {	expect -i $id timeout {} }]
    return [expr ! $result]
}

proc close_process {id} {
    # effects: if id is a valid spawn id, closes the connection to it (and
    #          performs a wait). Returns 1 in this case. Otherwise returns 0

    if {![is_valid_spawnid $id]} {
	return 0
    }
    # This catch statement is to prevent the race condition that
    # happens if the process has died by now
    set result [catch {
	close -i $id
	# wait -nowait -i $id
	wait -i $id
    } errorstring]
    if {$result} {
	send_user "Note: Process died after validity of spawn_id was determined: $errorstring\n"
	return 1
    }
}


# ------------------------ End of Procedure definitions -----------------



# Set up cleaning if process ends in the middle
exit -onexit thor_cleanup

# Useful regular expressions
set num_regexp "(\[0-9\]+)"
set till_eol "\[^\r\]*\r\n"
set FE_NAME_REGEXP "(FE id = $till_eol)"

# Set the default values
set OR_SOURCE ../or
set TOP "$env(HOME)/thor/alpha"
set hostname [exec hostname]
set portnum 0
# Do not create OO7 by default
set create_oo7 0
# Disable verbose outputs to the user by default
set verbose 0
log_user 0

# Make the timeout infinity so that the expect calls block
set timeout -1

# Make sure the program is started from the correct directory
if ![file exists ./thortest.tcl] {
    send_error "thortest.tcl: must be started from within the tests directory\n"
    exit 1
}

if ![file exists $OR_SOURCE/init/or.tcl] {
    send_error "thortest.tcl: could not find OR initialization files\n"
    exit 1
}

# Process command line paramters. See procedure "usage" for their meanings
set result [catch {
    set len [llength $argv]
    for {set i 0} {$i < $len} {incr i} {
	set option [lindex $argv $i]
	# send_user "Option: $len <$option>\n"
	if {![string compare "-v" $option]} {
	    log_user 1
	    set verbose 1
	} elseif {![string compare "-V" $option]} {
	    exp_internal 1
	} elseif {![string compare "-c" $option]} {
	    incr i
	    set value [lindex $argv $i]
	    if ![string compare "oo7" $value] {
		set create_oo7 1
	    } else {
		usage "invalid option \"$value\" supplied to -c flag"
	    }
	} elseif {![string compare "-p" $option]} {
	    incr i
	    set portnum [lindex $argv $i]
	} elseif {![string compare "-d" $option]} {
	    incr i
	    set TOP [lindex $argv $i]
	} else {
	    usage "unknown option \"$option\""
	}
    }
} errorstring]

if {$result} {
    send_user "Error: $errorstring\n"
    usage
}

# Check various optionss for reasonable values
if ![regexp {^[0-9]+$} $portnum] {
    usage "non-numeric value \"$portnum\" supplied to -p flag"
}
if ![file exists $TOP/or/or] {
    usage "OR executable not found in \"$TOP/or\""
}
if ![file exists $TOP/fe/fe] {
    usage "FE executable not found in \"$TOP/fe/fe\""
}
if ![file exists $TOP/client/C++/tests/dirTest] {
    usage "dirTest executable not found in \"$TOP/client/C++/tests\""
}
if ![file exists $TOP/client/C++/tests/inquirTest] {
    usage "INQUIR executable not found in \"$TOP/client/C++/tests\""
}
if ![file exists $TOP/client/C++/OO7/oo7] {
    usage "OO7 executable not found in \"$TOP/client/C++/OO7\""
}
if ![file exists $TOP/client/C++/tests/do_param_test] {
    usage "param_param_test executable not found in \"$TOP/client/C++/tests\""
}

# Derived variables set here 
set env(THOR) "$hostname:$portnum"
set env(PATH) "$TOP/or:$TOP/client/C++/tests:$TOP/client/C++/OO7:$TOP/fe:$env(PATH)"
send_user "TOP dir: $TOP\nVerbose: $verbose\n\
           \bCreate OO7: $create_oo7\nOR: $hostname:$portnum\n"

# Add calls to individual tests here. Whenever a new test suite is added
# for a different database, just add a new "source" statement here
# The requirement from each sourced file is that it does not leave any open
# processes at the end
# XXX What about global variables used in these files. Name clashes ??

# Make the FE shrink after every GC operation
set env(SHRINK_EVERY_GC) 1
