/*
\section{FE OR Communication Protocol}

This file describes the FE--OR communication protocol.
The overall strategy is to set-up a TCP connection between the
FE and the OR and then send messages over that connection.

The first thing sent over the TCP connection is an "fe_num"
sent by the FE (see "fe_num.h").  After that, this protocol
kicks in.
*/

#ifndef _FE_OR_MSG_H
#define _FE_OR_MSG_H

#ifndef __cplusplus
#error This file should only be included by C++ programs.
#endif

#include "utils/basic.h"
#include "utils/array.h"
#include "utils/bits.h"
#include "common/oref.h"
#include "common/xref.h"
#include "common/prefetch.h"
#include "common/or_stat.h"

class Network;

/*
We assign OR port numbers starting at a fixed number "OR_DEFAULT_FE_PORT".

OR ":0" gets assigned "OR_DEFAULT_FE_PORT".  OR $":"n$ gets assigned
"OR_DEFAULT_FE_PORT" + $n$.
*/
#define OR_DEFAULT_FE_PORT	4579

/* \subsection{Messages from the FE to the OR} */

#define FE_FETCHROOT	 101   /* Fetch the root (no data) */
#define FE_FETCH	 102   /* Fetch specified object (fe_fetch_request) */
// obsolete:
// #define FE_COMMIT        103   /* Commit transaction (fe_commit_request) */
#define FE_OBJECTS       104   /* Update fe_info at OR (fe_orefs) */
#define FE_INVALID_ACK   105   /* Invalidation message ack from FE */
#define FE_STAT		 106   /* Get OR performance information */
#define FE_PREPARE_COORD 107   /* Coordinator prepare (fe_commit_request) */
#define FE_PREPARE_PART  108   /* Participant prepare (fe_commit_request) */
#define FE_ROOT_REPLY    109   /* Reply to the OR volatile roots request */


// \subsubsection{Control Messages}
// These may be removed later, but are useful for performance tweaks.
#define FE_LOG_SIZE	151	/* Change log size */
#define	FE_CACHE_SIZE	152	/* Change cache size */
#define	FE_DSPACE_USE	153	/* Change disk space utilization */

// This is useful for testing & experiments,
// but shouldn't be used in real system
#define FE_START_GC       171   /* ask OR to start GC */
#define FE_CREATE_NEW_SEG 172   /* ask OR to create a new segment */

/* \subsubsection{Fetch Message} */
struct fe_fetch_request {
    Oref	  o;			/* Object to fetch */
    prefetch_hint prefetch;		/* Prefetching hints */
    int  clear_pref_set;                /* If true OR should clear the prefetch
                                           set for the requesting FE */
};

/* \subsubsection{Commit Message} */
struct fe_commit_request {
    Ubits32 dummy;          // Nothing for the moment.
    // After receiving FE_PREPARE_* , the transaction manager needs to call
    // Transaction::decode and retrieve the commit information from the
    // network.
};

/* \subsection{The {\tt FE\_OBJECTS} Message} */
struct fe_orefs {
    int num;       // number of orefs following
}; 

/* \subsection{The {\tt FE\_INVALID_ACK} Message} */
struct fe_invalid_ack {
    Ubits32 last_message_seen;    // Last invalidation message seen by FE
}; 


/* N of these are sent contiguously after an "fe_orefs" structure. */
struct fe_objdesc {
    Oref o;
};

/*
\subsubsection{The {\tt FE\_STAT} Message}
The FE can ask the OR for various performance characteristics by
sending an "FE_STAT" message.  The response will be an "OR_STAT"
message followed by an "or_stat" structure.  See "common/or_stat.h".
*/

/* \subsubsection{Possible FE to OR messages} */
struct fe_message {
    Ubits32 msgtype;
    union {
	fe_fetch_request  fetch;	/* FE_FETCH */
	fe_commit_request commit;	/* FE_COMMIT */
        fe_orefs          objects;      /* FE_OBJECTS */
        fe_invalid_ack    invalid_ack;  /* FE_INVALID_ACK */
	Ubits32		  log_size;	/* FE_LOG_SIZE: bytes */
	Ubits32		  cache_size;	/* FE_CACHE_SIZE: bytes */
	Ubits32		  dspace_use;	/* FE_DSPACE_USE: percent */
        Ubits32           partition;    /* FE_START_GC: partition */

	/* No data is sent with FE_FETCHROOT, FE_STAT */
    } u;

    bool encode(Network*) const;
    bool decode(Network*);
};

/* \subsection{Messages from the OR to the FE} */
#define OR_ROOT		201	/* Here is my root */
#define OR_FETCH_DENY	202	/* Fetch cannot be handled */
#define OR_OBJECTS	203	/* Bunch of objects */
#define OR_COMMITTED    204	/* The transaction was committed */
#define OR_STALEABORT   205	/* Transaction aborted - stale data*/
#define OR_FAILABORT    206	/* Timed out on some participant */
#define OR_INVALIDATION 208	/* Invalidation message for the FE */
#define OR_STAT		209	/* Response to "FE_STAT" message */
#define OR_READ_OK      210     /* Commit vote for read-only transaction */
#define OR_ROOT_REQUEST 211     /* Request for volatile roots from OR */
#define OR_NEW_SEG      212     /* Response to "FE_CREATE_NEW_SEG" message */

/*
\subsubsection{The {\tt OR\_OBJECTS} Message}

The message has three parts --

\begin{itemize}
\item a fixed size header
  This header contains N - the number of objects being sent over.

\item list of N object descriptors
  Foreach object descriptor, we give its oref, amount of
  dispatch header space allocated in the third part of the
  message, and amount of object data sent in the third part.

\item contents of the N objects
  Foreach object, we first have some slots of empty space
  where the FE can put dispatch headers, and then some slots
  that contain the actual data.  The sizes of these two
  regions are given in the descriptors sent in the second
  part of the message.
\end{itemize}
*/

/* Fixed size header. */
struct or_objects {
    Bits32   number;	/* Number of objects */
};

/*
Descriptor sent for each object.

N of these are sent (contiguously) after an "or_objects" structure.
*/

#define DEFAULT_NULLSPACE 1

struct or_objdesc {
    Oref o;		/* object reference */
    Bits32 objsize;	/* number of slots in real_obj */
};

// The following definition is useful in processing descriptor lists
declareArray(or_obj_descriptors,or_objdesc)

/*
Buffer with contents of N objects.  The ith object has
"DEFAULT_NULLSPACE" number of empty slots followed by desc[i].objsize
number of data slots.  The format of the data slots is given in
"or_obj.h".  Since this buffer is a completely variable-sized beast,
there is no sense in giving a C typedef for it here.
*/

/*
\subsection{The {\tt OR\_COMMITTED} Message}

The message has a header containing the number of new objects
made persistent by the transaction.  This header is followed
by two arrays.  One array is the list of xrefs assigned to
the new objects.  The second array is the list of uids assigned
to these new objects.
*/

struct or_commit {
    Bits32	count;		/* Number of new objects */
};

struct or_invalid {
    Ubits32	count;		/* Number of invalid objects */
    Ubits32     msg_start;      /* The range of invalidation message */
    Ubits32     msg_end;        /* numbers being sent in this mesage */
};

/* \subsubsection{Possible OR to FE Messages} */

struct or_message {
    Ubits32 msgtype;
    union {
	Oref		root;		/* OR_ROOT */
	Oref		denied;		/* OR_FETCH_DENY */
	or_objects	objects;	/* OR_OBJECTS */
	or_commit	commit;		/* OR_COMMITTED */
	or_invalid      invalid;        /* OR_INVALIDATION */
        int             segid;          /* OR_NEW_SEG */
    } u;

    bool encode(Network*) const;
    bool decode(Network*);
    bool decode_buf(void*, void**);
};

#endif /* _FE_OR_MSG_H */
