#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "utils/map.h"
#include "utils/fail.h"
#include "utils/hostname.h"
#include "locator.h"

/* \section{Locator implementation}

Eventually, when a name service is used instead of a file to determine OR
locations, these operations should change to contact the name server over the
network.
*/


// Instantiate the Hash table
#define IMPL_KEY Address
#define IMPL_VALUE Location
#include "utils/impl_map.t"

unsigned int const Max_line        = 100;  // Max. length of line in locator file

Locator::Locator() {
    lmap = new Location_map(2);
    if (!init())
	fail("Locator init failed");
}

Locator::~Locator() {
    delete lmap;
}

bool Locator::init() {
    char *fname;   // Location file
    char line[Max_line + 1], host_spec[50];
    FILE *infile;
    int or_num;

    fname = getenv("THORNAMES");
    if (fname == NULL) {
	// Construct default value for map if possible.
	if (init_default()) return TRUE;

	warn("The environment variables THOR, THORNAMES are both undefined");
	return FALSE;
    }

    // Read OR location file
    if ((infile = fopen(fname, "r")) == NULL) {
	warn("Can't open OR location file %s", fname);
	return FALSE;
    }

    bool result = TRUE;
    while (fgets(line, Max_line, infile) != NULL) {
	if (line[0] == '%' || sscanf(line, "%d %s", &or_num, host_spec) != 2)
	    continue;

	if (or_num <= 0) {
	    warn("%d: invalid OR number.  Must be positive.", or_num);
	    result = FALSE;
	    break;
	}

	if (!add_spec(or_num, host_spec)) {
	    result = FALSE;
	    break;
	}
    }

    fclose(infile);
    return result;
}

bool Locator::lookup(Address a, Ubits32 *host_id, int *port) {
    Location loc;
    bool found = lmap->find(a, loc);
    if (!found) return FALSE;
    *host_id = loc.addr;
    *port = loc.port;
    return TRUE;
}

void Locator::insert(Address a, Ubits32 host_id, int port) {
    Location loc;
    loc.addr = host_id;
    loc.port = port;
    th_assert(!lmap->contains(a), "Mapping already exists");
    lmap->store1(a, loc);
}

//
// Internal methods
//

bool Locator::init_default() {
    // Default initialization can occur only if THOR is defined.  The
    // default OR location map contains a single entry with OR number
    // 1 assigned to the OR named by the "THOR" environment variable.
    char const* spec = getenv("THOR");
    if (spec == 0) return FALSE;

#if !COMPILER
    warn("Running in single OR mode (THORNAMES not defined)");
#endif
    return (add_spec(1, spec));
}

bool Locator::add_spec(OR_num number, char const* spec) {
    struct sockaddr_in addr;
    if (!findport(spec, OR_DEFAULT_FE_PORT, &addr)) {
	warn("%s: invalid OR location", spec);
	return FALSE;
    }

    // Store the mapping from the or_address to the location
    Location loc;
    loc.addr = addr.sin_addr.s_addr;
    loc.port = ntohs(addr.sin_port);
    Address or_addr = OR_address(number);
    lmap->store1(or_addr, loc);
    return TRUE;
}
