/*
\section{Locator}

This abstraction provides a mapping from Addresses to <host_id, port>
pairs. Where host_id and port denote the host and port on which a process is
listening to connections.

This abstraction also exports default ports on which ORs and FEs listen for
new connections. For each FE and OR, there is a default port number and in the
environment variables, only the port offsets are specified.  The real ports
are genrated by adding these offsets to the default ports.

The mapping from OR numbers to locations is generated as follows:
  1. If environment variable THORNAMES is set, it is assumed to refer to a
     file that contains lines of the form
     OR_number host: port
     OR_number host: port
  2. Otherwise, the variable THOR must be set to host:port, which 
     is mapped from OR number 1.

This should be changed to use a highly available name service so that given
an OR number, its hostname and port can be identified.

*/

#ifndef _LOCATOR_H
#define _LOCATOR_H

#include "utils/basic.h"
#include "utils/bits.h"
#include "utils/address.h"
#include "utils/map.h"
#include "or_num.h"
#include "or_address.h"

// OR listens on this port for new connections from the FE by default
#define OR_DEFAULT_FE_PORT 4579
// FE listens on this port for new connections by default
#define FE_DEFAULT_FE_PORT 8000


// To keep track of the location of a single FE or OR
class Location {
public:
    Ubits32  addr;  // Internet address of process in network byte order
    int      port;  // Port on which the process listens
};

typedef Map<Address, Location> Location_map;

class Locator {
  public:

    Locator();
    //           Uses env variable THORNAMES and THOR. See header of file.
    //           Fails if initialization failed.

    ~Locator();

    bool lookup(Address a, Ubits32 *host_id, int *port);
    // modifies: host_id, port
    // effects:  Looks up "a", sets host_id and port to be the ports
    //           on which the process corresponding to "a" listens on

    void insert(Address a, Ubits32 host_id, int port);
    // requires: "a" is a process on "host_id" that listens for open
    //            connections on port "port" 
    //            No mapping for a exists
    // effects: Adds a mapping from a to <host_id, port>


  private:
    // Representation

    Location_map       *lmap; // Maps from address to IP address + port number

    bool init();
    // modifies: this
    // effects:  Initializes the locator "this".  
    //           Uses env variable THORNAMES and THOR. See header of file.
    //           Returns TRUE iff initialization succeeded.

    bool init_default();
    // effects	Initialize default value based on value of THOR environment
    //		variable.  Return TRUE iff successful.

    bool add_spec(OR_num number, char const* spec);
    // effects	Attempt to parse "spec" as an OR location and introduce
    //		a mapping "number->location" if successful.  Return
    //		TRUE iff "spec" was parsed successfully.
};

#endif /* _LOCATOR_H */
