#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "utils/compat.h"
#include <sys/uio.h>

#include "utils/array.h"
#include "utils/intarray.h"
#include "utils/device.h"
#include "utils/th_assert.h"
#include "slot.h"
#include "oref.h"
#include "or_obj.h"
#include "modset.h"

extern "C" {
  struct iovec;  
};

Modset::Modset() {
    num    = 0;
    buffer = new OR_slot_array;
    orefs  = new Orefs;
    sizes  = new IntArray;
}

Modset::~Modset() {
    delete buffer;
    delete orefs;
    delete sizes;
}

Obj_handle Modset::add_object(Oref oref, OR_obj* obj, int num_slots,
			      Oref class_oref) {
    Obj_handle handle =  buffer->size();
    buffer->_enlarge_by(OR_obj_headers + num_slots);
    OR_obj* rep = (OR_obj *) buffer->as_pointer() + handle;
    rep->set_class_oref(class_oref);     // Set the class_oref
    memcpy(rep->fields(), obj->fields(), Slot_size*num_slots);
    num++;

    orefs->append(oref);
    sizes->append(num_slots + OR_obj_headers);
    return handle;
}

void Modset::add_surrogate(Oref oref, OR_num dest_or, Oref dest_oref) {
    Obj_handle handle =  buffer->size();
    int total_slots = Surrogate_size()/Slot_size;
    buffer->_enlarge_by(total_slots);
    OR_obj* rep = (OR_obj *) buffer->as_pointer() + handle;
    // Create the surrogate in the buffer
    Surrogate_create(dest_or, dest_oref, rep);
    num++;
    orefs->append(oref);
    sizes->append(total_slots);
}

void Modset::set_oref(Obj_handle h, int slot, Oref oref) {
    buffer->slot(h + OR_obj_headers + slot).set_oref(oref);
}

// Format of header sent for ModSet.
struct Modset_net_header {
    Ubits32 o_num;		// Number of objects
    Ubits32 d_num;		// Number of data slots
};

int Modset::rep_size() const {
    return (sizeof(Modset_net_header) + Slot_size * buffer->size() +
	    sizeof(Oref) * orefs->size() + sizeof(Ubits32) * sizes->size());
}

bool Modset::encode(Device* dev) const {
    Modset_net_header h;
    h.o_num  = num;
    h.d_num  = buffer->size();

    // Set-up an io vector with 4 entries: header, data slots, orefs and sizes
    struct iovec iov[4];
    iov[0].iov_base = (caddr_t) &h;
    iov[0].iov_len  = sizeof(h);
    iov[1].iov_base = (caddr_t) buffer->as_pointer();
    iov[1].iov_len  = Slot_size * h.d_num;
    iov[2].iov_base = (caddr_t) orefs->as_pointer();
    iov[2].iov_len  = sizeof(Oref) * h.o_num;
    iov[3].iov_base = (caddr_t) sizes->as_pointer();
    iov[3].iov_len  = sizeof(Ubits32) * h.o_num;

    return (dev->send_vector(iov, 4));
}

bool Modset::decode(Device* dev) {
    Modset_net_header h;

    // Get sizes of various arrays and make enough space in the arrays
    if (! dev->recv_buffer(&h, sizeof(h))) return FALSE;
    num = h.o_num;

    buffer->clear();
    orefs->clear();
    sizes->clear();
    buffer->_enlarge_by(h.d_num);
    orefs->_enlarge_by(h.o_num);
    sizes->_enlarge_by(h.o_num);

    // Now set-up an io vector for reading the 3 arrays
    struct iovec iov[3];
    iov[0].iov_base = (caddr_t) buffer->as_pointer();
    iov[0].iov_len  = sizeof(OR_slot) * h.d_num;
    iov[1].iov_base = (caddr_t) orefs->as_pointer();
    iov[1].iov_len  = sizeof(Oref) * h.o_num;
    iov[2].iov_base = (caddr_t) sizes->as_pointer();
    iov[2].iov_len  = sizeof(Ubits32) * h.o_num;

    return (dev->recv_vector(iov, 3));
}

void Modset::clear() {
    num = 0;
    buffer->clear();
    orefs->clear();
    sizes->clear();
}

void Modset::print(bool whole_print, FILE* fp) {
    if (!fp) fp = stderr;
    int count = 0;

    OR_obj* o;
    Oref oref;
    int num_slots;
    Modset::Elements gen(this);

    while (gen.get(o, oref, num_slots)) {
	char str[40];
	Oref_unparse(oref, str);
	fprintf(fp, "Oref = %s, Size = %d\n", str, num_slots+OR_obj_headers);
	if (whole_print) o->print(num_slots, fp);
	count++;
    }
}
