/*
\section{OR set implementation}

The representation is just an unsorted array, but since we expect 
the number of participants to be small, it should be sufficiently fast.
*/

#include "utils/device.h"
#include "or_set.h"
#include "utils/arraysort.h"

implementArray(OR_array, OR_num)

static int OR_num_compare(void const *o1, void const *o2);

OR_set::OR_set() {
}

OR_set::~OR_set() {
    set.clear();
    set.reclaim();
}

OR_set& OR_set::operator=(OR_set const& old_set) {
    set = old_set.set; 
    return *this;
}


void OR_set::add(OR_num or_num) {
    if (!member(or_num))
	set.append(or_num);
}

bool OR_set::remove(OR_num or_num) {
    int i, size = set.size();

    for (i = 0; i < size; i++)
	if (set.slot(i) == or_num)
	{
	    OR_num last = set.remove();
	    if (i != size - 1)
		set.slot(i) = last;
	    return TRUE;
	}
    
    return FALSE;
}

bool OR_set::member(OR_num or_num) {
    int i, size = set.size();

    for (i = 0; i < size; i++)
	if (set.slot(i) == or_num)
	    return TRUE;
    return FALSE;
}

void OR_set::clear() {
    set.clear();
}

int OR_set::size() {
    return set.size();
}

int OR_set::encode_size() const {
    return sizeof(Ubits32) + (set.size() & sizeof(OR_num));
}

bool OR_set::encode(Device *dev) const {
    // First write out # of elements, and then the elements themselves.
    Ubits32 size = set.size();

    if (!dev->send_ubits32(size, TRUE))
	return FALSE;

    return dev->send_buffer(set.as_pointer(), size * sizeof(OR_num), TRUE);
}

bool OR_set::decode(Device *dev) {
    // First read number of ORs, and then OR numbers themselves.
    Ubits32 size;

    if (!dev->recv_ubits32(&size))
	return FALSE;

    set.clear();
    set._enlarge_by(size);

    return dev->recv_buffer(set.as_pointer(), size * sizeof(OR_num));
}

void OR_set::print(FILE *fp) {
    int i;
    if (!fp) fp = stderr;

    for (i=0; i < set.size(); i++)
	fprintf(fp, "%d ", set.slot(i));
    fprintf(fp, "\n");
}

// \subsection{Element generators}

OR_set::Elements::Elements(OR_set const* s) {
    index = 0;
    list = &s->set;
}

bool OR_set::Elements::get(OR_num& or_num) {
    if (index >= list->size()) return FALSE;
    or_num     = list->slot(index);
    index++;
    return TRUE;
}

OR_set::SortedElements::SortedElements(OR_set const* s) {
    index = 0;
    ArraySort(s->set, OR_num_compare);
    list = &s->set;
}

bool OR_set::SortedElements::get(OR_num& or_num) {
    if (index >= list->size()) return FALSE;
    or_num     = list->slot(index);
    index++;
    return TRUE;
}

static int OR_num_compare(void const *o1, void const *o2) {
    return *((OR_num *) o1) - *((OR_num *) o2);
}
