#!/usr/local/bin/expect --

# Note: All globals are prefixed by a G_

source useful.tcl

proc usage {{reason {}}} {
    if [string compare $reason {}] {
	send_error -- "perf.tcl: $reason\n\n"
    }

    send_error {Usage: perf.tcl [-v] [-V] [-s] [-e] [-c oo7] [-p port] [-d dir]
	[-r raw] [-i iterations] [-o or_machine] [-f fe_machine] [-m]
	[-db dbfile] [-fcache fe_cachesize] [-ocache or_cachesize]
	[-temp temp] [-trav travlist] [-W dynamic | static] [-h count]
	[-fargs args] [-oargs args] [-odat file] [-fdat file] [-name dbname]
	[-hotcold static | dynamic]
	-v                : Verbose output (default off)
	-V                : Extremely verbose (default off)
	-W dynamic|static : Shifting workload: dynamic or static (def: none)
	-hotcold static | : Hotcold workload: static or dynamic [def: none]
                 dynamic
	-m                : Multiple transactions for hot runs [default: off]
	-p <port>         : Server port number (default = 0)
	-db <dbfile>      : Name of the database [default: /tmp/or_$user]
	-d <directory>    : Top level directory for thor executables
			    (default TOP = $HOME/thor/alpha)
			    Database is searched in TOP/or
	-i <iterations>   : Iterations (reruns) for each exper [Default = 3]
        -o <machine name> : Machine to use for the OR
	-or <or_exec>     : Name of the or executable [default: or]
        -ocache <osize>   : Size of the fe cache in Kbytes [default: 30720]
	-oargs <args>     : Other arguments to the OR (multiple -oargs allowed)
	-odat file        : File that contains OR output [def: /tmp/$user.or]
        -s                : Print on the screen also [default: off]
	-e                : Not check that FE/OR already exists [def: check]
	-f <machine name> : Machine to use for the FE
	-fe <fe_exec>     : Name of the fe executable [default: fe]
        -fcache <fsize>   : Size of the fe cache in Kbytes [default: 12288]
	-fargs <args>     : Other arguments to the FE (multiple -fargs allowed)
	-fdat file        : File that contains FE output [def: /tmp/$user.fe]
	-r raw            : use raw partitions
	-temp <temp>      : Temperature of the exper(hot/cold) [def: cold]
        -trav <travlist>  : List of traversals [default: T6, T1, T2a, T2b]
	                  : For dynamic/hotcold wloads, it is list of (t6 t1 t3)
	-h <repeat count> : No. of times the hot traversal is executed [def:2]
	-name dbname      : Name of db (small/medium/shifting) [def: small]
    }
    send_error "\n"
    exit
}

proc spawn_or {or_executable name machine_name portnum args or_fileid} {
    # requires: Db "name" has been created already and or_fileid is a valid
    #           file id. or_Executable is the name of the server executable
    # effects: Start a new or with the database "$name" on
    #          "machine_name:$portnum". If it fails, it  keeps trying with
    #          higher port numbers. It passes args to the or and writes
    #          the output to or_fileid

    global G_TOP env G_hostname G_check_existing_process

    set sent_portnum $portnum
    set env(THOR) "$machine_name:$portnum"

    if {$G_check_existing_process && [process_exists or $machine_name]} {
	send_user "Server exists: OR not being spawned (try with -e)\n"
	exit -1
    }
    set args_string [join $args " "]

    set trials 0
    set timeout 10
    while 1 {
	incr trials
	set or_args "$args_string $name"
	set or_command [concat $or_executable $or_args]
	send_file $or_fileid 1 "OR Command: $or_command\n"

	# Also check that a OR does not already exist
	
	if {[string compare $G_hostname $machine_name]} {
	    spawn rsh $machine_name "setenv THOR $machine_name:$portnum;\
                $or_executable $or_args"
	} else {
	    eval spawn "$or_executable $or_args"
	}
	
	set done 0
	# Wait for the server to start up or start a new one if this dies
	while 1 {
	    set result [catch {
		expect "Server ready" {set done 1}
	    } errorstring]
	    
	    if {$result || $done} break;
	}
	if {$done} {break;}
	
	if {$trials >= 6} {
	    send_error "could not start OR: tried $trials times\n"
	    exit 1
	}

	close_process $spawn_id
	incr portnum
	set env(THOR) "$machine_name:$portnum"
    }
    if {$portnum != $sent_portnum} {
	send_user "Note: Port number $portnum chosen\n"
    }
    return $spawn_id
}

proc thor_cleanup {} {
    # effects: Called whenever the test file exits
    # puts stderr "Exited from test file\n"

    # Try to kill the running ORs and FEs
    global G_or_machine G_fe_machine G_fe_executable G_or_executable
    global G_or_fileid G_fe_fileid G_files_open

    send_user "Destroying OR on $G_or_machine and FE/OO7 on $G_fe_machine ... "

    kill_process $G_or_machine $G_or_executable
    kill_process $G_fe_machine $G_fe_executable
    kill_process $G_fe_machine oo7
    send_user "done\n"

    set result [catch {
	close $G_or_fileid
	close $G_fe_fileid
    } errorstring]

    if {!$result} {
	send_user "Statstics files closed\n"
    } else {
	send_user "Error while closing FE/OR files: $errorstring\n"
    }
}

proc spawn_client {client_machine or_id or_fileid fe_fileid client_dir
    client_name fe_args c_args fe_id client_id fe_name} {

    # requires: client_name refers to a valid client executable
    #           c_args are the arguments to be passed to the client. They
    #           should not specify the FE port/shared memory region explicitly
    #           or_id refers to the OR this FE will talk to
    #		or_fileid refers to the id of the file where the OR results
    #           are written
    #           fe_args are the arguments to the FE
    # modifies: fe_id client_id fe_name
    # effects: Spawn a client process as specified by client_str. Put the
    #          spawn_id of the client and the FE in fe_id and client_id
    #          Sets fe_name to be the fe_name nformation printed out by FE
    #          machine_name gives the name where the client is to be spawned
    #          client_machine gives the name where the client is to be spawned
    #          fe_fileid is the file where the output gets written to for FE
    
    upvar $fe_id FE_ID
    upvar $client_id CLIENT_ID
    upvar $fe_name FE_NAME
    global G_FE_NAME_REGEXP env G_fe_machine G_check_existing_process
    global G_TOP G_hostname G_fe_executable

    if {$G_check_existing_process && [process_exists fe $G_fe_machine]} {
	send_user "Front End exists: FE not being spawned (try with -e)\n"
	exit -1
    }

    global env
    set env(FE_FLAGS) $fe_args
    set env(FE_PROGRAM) $G_fe_executable

    # XXX Currently not spawning OO7 but simply spawning the FE

    if {[string compare $G_hostname $client_machine]} {
	spawn rsh $client_machine "setenv THOR $env(THOR);\
        setenv PROFDIR $env(PROFDIR); setenv FE_FLAGS \" $env(FE_FLAGS)\";\
        setenv FE_PROGRAM $env(FE_PROGRAM);\
        setenv PATH .; $G_fe_executable $env(FE_FLAGS)"
    } else {
	eval spawn "$G_fe_executable $env(FE_FLAGS)"
    }

    send_file $fe_fileid 1 "FE Command: $G_fe_executable $env(FE_FLAGS)\n"

    set CLIENT_ID $spawn_id
    set FE_ID $CLIENT_ID

    expect -i $FE_ID -re "FE configuration:"
    while {1} {
	expect {
	    -i $FE_ID -re $G_FE_NAME_REGEXP {
		send_file $fe_fileid 1 $expect_out(buffer);
		set FE_NAME $expect_out(1,string)
		break;
	    }
	    -i $FE_ID -re ".*\r\n" {
		send_file $fe_fileid 1 $expect_out(buffer);
	    }
	}
    }
    # regsub -all "\r|\n" $FE_NAME "" FE_NAME
    expect -i $or_id $FE_NAME
    # expect -i $or_id -gl "OR STATISTICS END"
}

proc spawn_oo7 {c_mach or_id or_fileid fe_fileid fe_args c_args fe_id
    client_id fe_name} {
    # modifies: fe_id client_id fe_name
    # effects:  Spawn the OO7 client with the c_args for oo7 and 
    #           fe_args for fe.
    # effects: Spawn the OO7 client as specified

    upvar $fe_id FE_ID
    upvar $client_id CLIENT_ID
    upvar $fe_name FE_NAME
    global env

    spawn_client $c_mach $or_id $or_fileid $fe_fileid OO7 oo7 $fe_args \
    $c_args FE_ID CLIENT_ID FE_NAME
    # Wait for the command string

    expect -i $CLIENT_ID -re "Enter run type:"
}

proc print_traversal_data {or_id or_fileid fe_fileid client_id traversal hot
                           multi fe_name} {
    # requires: A traversal given by $traversal has been started on $client_id
    # effects: Print the traversal data from the client and or.

    global G_fe_machine G_or_machine G_fe_executable G_or_executable
    set field_re "(\[a-zA-Z0-9 \]+)"
    set value_re "(\[0-9\.\]+)"
    set traverse_time -1.0
    set commit_time -1.0

    # Print out the OR id, FE id info and the traversal run info

    # Get the or pid and fe pid
    set or_pid [process_exists $G_or_executable $G_or_machine]
    set fe_pid [process_exists $G_fe_executable $G_fe_machine]
    # th_assert [expr $or_pid && $fe_pid] "OR or FE pid is 0, FE($fe_pid), OR($or_pid)"

    global G_to_screen G_shifting_workload G_hotcold_workload
    set to_revert 0
    if {!$G_to_screen} {
	set G_to_screen 1
	set to_revert 1
    }
    
    set out_data "\nTrav: T$traversal, Hot: $hot, Multi: $multi, Shift:\
               $G_shifting_workload, HCold = $G_hotcold_workload\n\
               OR (Pid): $or_pid, FE (Pid) = $fe_pid\n\n"
    send_file $fe_fileid 1 $out_data

    if {$to_revert} {
	set G_to_screen 0
    }
    send_file $or_fileid 1 $out_data

    # Get the traversal data first
    expect -i $client_id -re "Traversal STARTED\r\n"

    while {1} {
	expect {
	    -i $client_id -re "Traversal DONE\r\n" { 
		send_file $fe_fileid 1 $expect_out(buffer)
		break;
	    }
	    -i $client_id -re ".*\r\n" {
		send_file $fe_fileid 1 $expect_out(buffer);
	    }
	}
    }
    # Now get the FE data
    send -i $client_id "D\n"
    expect -i $client_id -re "Statistics STARTED"

    while {1} {
	expect {
	    -i $client_id -re "Statistics DONE\r\n" { 
		send_file $fe_fileid 1 $expect_out(buffer);
		break;
	    }
	    -i $client_id -re ".*\r\n" {
		send_file $fe_fileid 1 $expect_out(buffer)
	    }
	}
    }
		
    if {![string compare $G_or_machine $G_fe_machine]} {
	send_user "Error: FE on the same machine\n"
	exit -1
    }
    # Now kill the fe and get the OR data
    send_user "Killing all the FE/OO7 processes on $G_fe_machine ... "

    send -i $client_id "q\n"
    sleep 1
    close_process $client_id
    kill_process $G_fe_machine $G_fe_executable
    kill_process $G_fe_machine oo7
    send_user "done\n";

    # expect -i $or_id -gl "OR STATISTICS END"
    # send_file $or_fileid 1 "$expect_out(buffer)\n";

    expect -i $or_id -gl "FE DIED $fe_name"
    send_file $fe_fileid 1 "\n"
}

proc do_traversal {or_machine client_machine or_fileid fe_fileid or_args
    fe_args c_args traversal hot multi} {
    # effects: Start traversal (1, 2a, 2b etc) with a new OR
    #          Booleans hot and multi indicate whether the traversal is hot
    #          and multiple transactions need to be committed for traversal
    #          or_machine specifies the machine where the OR is to be run
    #          Kills off the OR and client after it is done

    set total_ttime 0
    set total_ctime 0
    global G_iters G_run_number G_iter_num G_exp_number
    global G_linestr G_hot_count

    set out_line "$G_linestr\n"
    send_file $or_fileid 1 $out_line
    set out_data "EXECUTION NUM = $G_exp_number,\
               Run = $G_run_number, Iter = $G_iter_num STARTED \n"
    send_file $fe_fileid 1 $out_data
    send_file $or_fileid 1 $out_data
    send_file $fe_fileid 1 $out_line
    send_file $or_fileid 1 $out_line

    if {$hot} {
	if {$multi} {
	    set hotinfo "H\nM\n$G_hot_count"
	} else {
	    set hotinfo "H\nS\n$G_hot_count"
	}
    } else {
	set hotinfo "C"
    }

    global G_DB_FILENAME G_DB_NAME G_or_executable
    global G_portnum G_shifting_workload G_hotcold_workload

    set or_id [spawn_or $G_or_executable $G_DB_FILENAME $or_machine \
		   $G_portnum $or_args $or_fileid]

    spawn_oo7 $client_machine $or_id $or_fileid $fe_fileid $fe_args $c_args \
	fe_id client_id fe_name

    set trav_command "T"
    set trav_specific "$traversal\n$hotinfo"

    if {![string compare "static" $G_hotcold_workload]} {
	set trav_command "H"
	set trav_specific "N\n$traversal"
    } elseif {![string compare "dynamic" $G_hotcold_workload]} {
	set trav_command "H"
	set trav_specific "S\n$traversal"
    }

    if {![string compare "static" $G_shifting_workload]} {
	set trav_command "K"
	set trav_specific "S\n$traversal"
    } elseif {![string compare $G_shifting_workload "dynamic"]} {
	set trav_command "K"
	set trav_specific "D\n$traversal"
    }

    set final_command "M\n$trav_command\n$G_DB_NAME\n$trav_specific\n"
    send -i $client_id 	$final_command
    regsub -all "\n" $final_command "," pr_final_command
    
    set out_data "Final command = $pr_final_command, "
    send_file $fe_fileid 1 $out_data

    set results [print_traversal_data $or_id $or_fileid $fe_fileid $client_id \
		     $traversal $hot $multi $fe_name]

    # Now destroy the OR process. Wait for log drain

    send_user "Killing all the or processes on $or_machine ... "
    close_process $or_id
    kill_process $or_machine $G_or_executable
    send_user "done\n"

    send_file $fe_fileid 1 "\n[exec date]\n"
    send_file $fe_fileid 1 "EXECUTION NUM = $G_exp_number,\
               Run = $G_run_number, Iter = $G_iter_num DONE \n\n"
    global G_linestr
    send_file $fe_fileid 1 "$G_linestr\n"
    incr G_run_number
}

# ------------------------ End of Procedure definitions -----------------

# Set up cleaning if process ends in the middle
exit -onexit thor_cleanup

# Useful regular expressions
set G_FE_NAME_REGEXP "(Id = ($till_end))"

# Set the default values
# set G_TOP "$env(HOME)/thor/optim"
set G_TOP "$env(HOME)/thor/alpha"
set env(PROFDIR) "$G_TOP/prof"
set G_or_machine "harmony"
set G_portnum 0
set G_run_number 0
# Disable verbose outputs to the user by default
set G_fe_machine "clef"
set use_raw 0
set G_check_existing_process 1
set G_iters 1
set G_fe_execname "fe"
set G_or_execname "or"
set G_DB_FILENAME "/tmp/or_$username"
set G_DB_NAME "small"
set fe_cachesize "12288"
set or_cachesize "30720"
set G_exp_temp "cold"
set G_travlist "6 1 2b 2a"
set G_fe_executable "$G_TOP/fe/$G_fe_execname"
set G_or_executable "$G_TOP/or/$G_or_execname"
set G_shifting_workload 0
set G_hotcold_workload 0
set G_hot_count 2
set G_multi_trans 0
set G_or_filename "/tmp/$username.or"
set G_fe_filename "/tmp/$username.fe"
set G_fe_args ""
set G_or_args ""

set result [catch {
    set len [llength $argv]
    for {set i 0} {$i < $len} {incr i} {
	set option [lindex $argv $i]
	# send_user "Option: $len <$option>\n"
	if {![string compare "-v" $option]} {
	    log_user 1
	    set verbose 1
	} elseif {![string compare "-V" $option]} {
	    exp_internal 1
	} elseif {![string compare "-e" $option]} {
	    set G_check_existing_process 0
	} elseif {![string compare "-s" $option]} {
	    set G_to_screen 1
	} elseif {![string compare "-m" $option]} {
	    set G_multi_trans 1
	} elseif {![string compare "-W" $option]} {
	    incr i
	    set G_shifting_workload [lindex $argv $i]
	    if {[string compare "dynamic" $G_shifting_workload] &&
		[string compare "static" $G_shifting_workload]} {
		usage "invalid option \"$G_shifting_workload\" supplied to\
                       -W flag"
	    }
	} elseif {![string compare "-hotcold" $option]} {
	    incr i
	    set G_hotcold_workload [lindex $argv $i]
	    if {[string compare "dynamic" $G_hotcold_workload] &&
		[string compare "static" $G_hotcold_workload]} {
		usage "invalid option \"$G_hotcold_workload\" supplied to\
                       -hotcold flag"
	    }
	} elseif {![string compare "-h" $option]} {
	    incr i
	    set G_hot_count [lindex $argv $i]
	} elseif {![string compare "-r" $option]} {
	    incr i
	    set value [lindex $argv $i]
	    if ![string compare "raw" $value] {
		set use_raw 1
	    } else {
		usage "invalid option \"$value\" supplied to -r flag"
	    }
	} elseif {![string compare "-f" $option]} {
	    incr i
	    set G_fe_machine [lindex $argv $i]
	} elseif {![string compare "-o" $option]} {
	    incr i
	    set G_or_machine [lindex $argv $i]
	} elseif {![string compare "-fe" $option]} {
	    incr i
	    set G_fe_execname [lindex $argv $i]
	} elseif {![string compare "-or" $option]} {
	    incr i
	    set G_or_execname [lindex $argv $i]
	} elseif {![string compare "-fcache" $option]} {
	    incr i
	    set fe_cachesize [lindex $argv $i]
	} elseif {![string compare "-ocache" $option]} {
	    incr i
	    set or_cachesize [lindex $argv $i]
	} elseif {![string compare "-fargs" $option]} {
	    incr i
	    set G_fe_args "$G_fe_args [lindex $argv $i]"
	} elseif {![string compare "-oargs" $option]} {
	    incr i
	    set G_or_args "$G_or_args [lindex $argv $i]"
	} elseif {![string compare "-fdat" $option]} {
	    incr i
	    set G_fe_filename [lindex $argv $i]
	} elseif {![string compare "-odat" $option]} {
	    incr i
	    set G_or_filename [lindex $argv $i]
	} elseif {![string compare "-temp" $option]} {
	    incr i
	    set G_exp_temp [lindex $argv $i]
	} elseif {![string compare "-trav" $option]} {
	    incr i
	    set G_travlist [lindex $argv $i]
	} elseif {![string compare "-p" $option]} {
	    incr i
	    set G_portnum [lindex $argv $i]
	} elseif {![string compare "-db" $option]} {
	    incr i
	    set G_DB_FILENAME [lindex $argv $i]
	} elseif {![string compare "-dbname" $option]} {
	    incr i
	    set G_DB_NAME [lindex $argv $i]
	} elseif {![string compare "-i" $option]} {
	    incr i
	    set G_iters [lindex $argv $i]
	} elseif {![string compare "-d" $option]} {
	    incr i
	    set G_TOP [lindex $argv $i]
	} else {
	    usage "unknown option \"$option\""
	}
    }
} errorstring]

if {$result} {
    send_user "Error: $errorstring\n"
    usage
}

# Check various options for reasonable values
if {![string compare $G_or_machine $G_fe_machine]} {
    send_user "FE and OR machines are the same\n"
    exit 1
}

# Check that the traverse probabilities are ok
if {![string compare $G_shifting_workload "dynamic"] ||
    [string compare $G_hotcold_workload "0"]} {
    foreach travprob $G_travlist {
	if ![regexp {^[0-9]+ [0-9]+ [0-9]+$} $travprob] {
	    usage "Bad probability list $travprob"
	}
    }
}

if ![regexp {^[0-9]+$} $G_portnum] {
    usage "non-numeric value \"$G_portnum\" supplied to -p flag"
}

#if ![file exists $G_TOP/client/C++/OO7/oo7] {
#    usage "OO7 executable not found in \"$G_TOP/client/C++/OO7\""
#}

# Derived variables set here 
set env(THOR) "$G_or_machine:$G_portnum"
set env(PATH) "$G_TOP/or:$G_TOP/client/C++/OO7:$G_TOP/fe:$env(PATH)"
set G_fe_executable "$G_TOP/fe/$G_fe_execname"
set G_or_executable "$G_TOP/or/$G_or_execname"
set G_fe_args "-s $fe_cachesize $G_fe_args"
set G_or_args "-S $or_cachesize $G_or_args"
