#ifndef _Compacted_map_h
#define _Compacted_map_h 1

#include "utils/th_assert.h"
#include "common/or_num.h"
#include "common/oref.h"
#include "utils/hashfuncs.h"

class Compacted_map {
  //
  // Overview: Map from <or,page_id> to number of objects from page <or,page_id> 
  //   in compacted cache frames.
  //
  // Invariant: The number of objects is greater than zero for all mappings.
  //
public:
  Compacted_map(unsigned size);
  // Effects: Creates a compacted page map. size should be the expected
  //   size of the map for good performance.

  ~Compacted_map();
  // Effects: Destroys the map and all its associated storage.

  int increment(OR_num orx, Uint page_id, Uint delta);
  // Requires: delta > 0 
  // Effects: If there is a mapping for page <or, page_id>, adds
  //   delta to its count; otherwise, creates an entry with count delta.
  //   Returns the resulting count for the entry.

  int decrement(OR_num orx, Uint page_id, Uint delta);
  // Effects: If there is a mapping "m" for page <or, page_id> in
  //   this: it decrements the count in "m" by delta; if the resulting
  //   count is less than or equal to zero the entry is deleted; and
  //   returns the resulting count. If there is no entry for the page
  //   returns 0.

  int lookup(OR_num orx, Uint page_id);
  // Effects: Looks up the page <or, page_id> in the map. If found returns
  //   the number of objects from the page in compacted frames in the cache.
  //   Otherwise, returns 0.

  void print_stats(ostream &o);
  // Effects: Prints map statistics.
  
private:
  
  class Page_mapping {
    //
    // Overview: Describes a compacted page in the persistent cache.
    //   It contains the pages's or and page_id, and counter with number
    //   of objects in the cache.
    //
  public:
    Page_mapping() { or_ = 0; page_id_count = 0; }
    // Effects: Creates a compacted page entry.
  
    unsigned page_id() { return page_id_count >> Onum_bits; }
    // Effects: Returns the page id.

    void set_page_id(unsigned page_id) {
      // Effects: Sets the page id.
      th_assert(page_id < (1 << Page_id_bits), "Invalid argument\n");
      page_id_count = (page_id_count & Onum_bitmask) | (page_id << Onum_bits);
    }

    OR_num orx() { return or_; }
    // Effects: Returns the or of this entry.
    
    void set_or(OR_num o) { or_ = o; }
    // Effects: Sets the OR number.

    int count() { return Onum_bitmask & page_id_count; }
    // Effects: Returns the count of objects in page.

    void set_count(int cnt) { 
      // Effects: Sets the count in the entry to be cnt.
      th_assert(cnt <= Max_onum, "Invalid argument\n");
      page_id_count = (page_id_count & ~Onum_bitmask) | cnt;
    }

    void mark_free() { page_id_count = 0; }
    // Effects: Marks this as free (hash table entry)

    bool is_free() { return page_id_count == 0; }
    // Effects: Returns true iff this as free (hash table entry)

  private:
    // Representation
    OR_num or_;                  // Page's or.
    unsigned page_id_count;      // Page id : count of objects in compacted frames.
  };


  struct Compacted_map_entry {
    // Overview: Groups several Page_entry and a shared next into
    //   a 32 byte cache line.
    Compacted_map_entry() {next = 0;}
    
    Page_mapping mappings[3];
    Compacted_map_entry *next;
  };


  Compacted_map_entry *table;   // Bucket table for the hash map.   
  unsigned size;              // Number of buckets in "buckets". Must be power of 2.
  unsigned shift_bits;        // Number of bits in an unsigned minus bits in size.
  unsigned num_entries;       // Number of entries allocated in map.
  unsigned num_mappings;      // Number of mappings in map.
};


inline int Compacted_map::lookup(OR_num orx, Uint page_id) {
  return decrement(orx, page_id, 0);
}

  
#endif // __Compacted_map_h



