
#include <string.h>
#include <stream.h>
#include <math.h>

#include "compiler/C++wrappers/fields.h"
#include "compiler/C++wrappers/core.h"

#include "runtime/or_info.h"
#include "runtime/stats.h"

#include "persistent_cache.h"
#include "scanned_set.h"
#include "compacted_map.h"
#include "resident_object_table.h"
#include "page_map.h"
#include "optimal_replacer.h"

#include "fe/main/fe_config.h"

#include "config/vdefs/OPTIMAL_PAGE_REPLACEMENT.h"
#include "config/vdefs/LRU_PAGE_REPLACEMENT.h"
#include "config/vdefs/USE_REF_COUNTS.h"

#include "utils/th_assert.h"
#include "utils/fail.h"
#include "utils/array.h"
#include "utils/mdebug.h"
#include "utils/Monitor.h"

#include "frame_info.h"

extern Compacted_page Invalid_compacted_map_entry;

// Constant used to avoid integer division by FE->discard_factor
const unsigned Discard_factor_bits = 7;


inline 
void Persistent_cache::survival_fraction(Uint &usage, Uint count, Uint num_survived) {
  // The following code computes a 2 bit approximation of the fraction 
  // num_survived/count. 
  unsigned count_half = count >> 1;
  unsigned count_fourth = count >> 2;
  usage |= (num_survived > count_half) << 1;
  usage |= (!(num_survived > count_half)) & (num_survived > count_fourth);
  usage |= (num_survived > count_half + count_fourth);
}


void Persistent_cache::usage_value(Uint &usage, int *usage_hist, Uint count) {
  Uint i;

  // Compute page usage:
  if (FE->use_eviction) {
    // Eviction: Page usage is the bitwise or of object usages.
    for (i = 0;  i < Max_usage+1; i++) 
      if (usage_hist[i] != 0) usage |= i;
    return;
  }

  //
  // Compaction:
  //
  // The page usage is threshold:4 | survival:2,
  // where threshold is the threshold value that ensures at least
  // 1/FE->discard_factor of the objects in the page is discarded,
  // and survival indicates what fraction of the objects in the
  // page survive compaction with threshold.
  //
  // Compute threshold:
  Uint cur_count = 0;
      
  // The next statement computes count/FE->discard_factor (without the divlu).
  Uint critical_count = (count*discard_multiplier) >> Discard_factor_bits;
  if (critical_count == 0) critical_count = 1;
  for (i = 0;  i < Max_usage+1; i++) {
    cur_count += usage_hist[i];
    if (cur_count >= critical_count) break;
  }

  // Set threshold portion of usage.
  usage |= (i << Extra_usage_bits);
  survival_fraction(usage, count, count - cur_count);
} 


inline unsigned Persistent_cache::collect_usage_and_decay(Uint page_index, bool decay) {
  // Breaking into the page abstraction

  register Frame_info *pinfo = page_frame_info+page_index;
  Page *p = page(page_index);
  unsigned usage = 0;
  register unsigned i = 0;
  register long decay_val = (decay) ? 1 : 0;

  // Histogram with decayed usages of objects in the page. Initially empty.
  int usage_hist[Max_usage+1];
  for (i=0; i < Max_usage+1; i++) usage_hist[i] = 0;
                
  // If page was not installed none of its objects could have been
  // used. Therefore, it is not scanned and its usage is 0.
  if (!pinfo->none_installed()) {
    // If page was installed scan its objects to compute the page's usage,
    // and at the same time decay the usage of each object.
    FE_STATS(num_pages_scanned++)
    register Slot *data = p->data;
    register Offset *offsets = p->header.offsets;
    register unsigned osize = p->header.osize;
    int null_count = 0;

    for (i=0; i < osize; i++) {
      if (pinfo->is_installed(i)) {
	// If object was installed decay object usage.
	Fields f = (Fields) (data + offsets[i]);
	Uint new_obj_usage = (f->get_usage() + decay_val) >> decay_val;
	f->set_usage(new_obj_usage);
	usage_hist[(f->is_written()) ? Max_usage : new_obj_usage]++;
      } else {
	if (offsets[i]) { 
	  // If offsets entry is non-null then object was not installed.
	  usage_hist[0]++;
	} else {
	  // If offsets entry is null then object does not exist.
	  null_count++;
	}
      } 
    }
    usage_value(usage, usage_hist, p->header.osize - null_count);
  }

#if COLLECT_EXPENSIVE_STAT
  for (i = 0; i <= Max_usage; i++) usage_buckets[i] += usage_hist[i];
#endif

  // Update page's decay status to note we decayed the usage of its objects.   
  if (decay) page_decay_status[page_index] = cache_decay_status;
 
  return usage;
}



void Persistent_cache::trace_replacement(int pages_to_discard) {
  FE_STATS(number_of_scans++)
  FE_STATS(page_scan_time.start())
  trace_rep->bring_up_to_date(Access_counter);
  FE_STATS(page_scan_time.stop())

  if (pages_to_discard > 0) {
    FE_STATS(page_compaction_time.start())

    Page *victim = trace_rep->replace_page();
    last_scanned->insert(index(victim), Max_usage);
    last_scanned->end_scan();

    pages_to_discard -= compact(pages_to_discard, true);
    FE_STATS(page_compaction_time.stop())

    // Current implementation of optimal replacement assumes at most
    // one page is evicted per fetch and there are no modified objects.
    th_assert(pages_to_discard == 0, "Optimal replacement assumption is not valid");
  }
}


void Persistent_cache::stack_scan() {
  register Bool_vector *pointed_pages = stack_refs;
  register Bool_vector *pointed_cores = FE->rot->get_stack_refs();
  
  // pointed_pages is cleared here because it is small and clearing it 
  // is inexpensive. pointed_cores is large and is not cleared here; It is cleared 
  // incrementally when it is checked (see resident_object_table.h).
  pointed_pages->clear();
   

  // Put register values into the stack and set up bottom and top of stack.
  jmp_buf env;      // variable that will hold the register values
  _setjmp(env);     // fill in register values should work in most
                    // versions of Unix.

  void **stack_now = (void**)&env;
  register void **stack_low;
  register void **stack_hi;
  if (stack_now < Stack_main) { // for portability
    stack_low = stack_now;
    stack_hi  = Stack_main;
  } else {
    stack_low = Stack_main;
    stack_hi  = stack_now + sizeof(env)/sizeof(void*);
  }

  register const void *start = store->start();
  register const Uint pc_size = store->size()*sizeof(Page);
#if USE_REF_COUNTS
  register const Uint tot_size =  pc_size+FE->rot->bytes();
#else
  register const Uint tot_size =  pc_size;
  Bool_vector *vec = pointed_pages;
#endif

  for (; stack_low < stack_hi; stack_low++) {
    Uint index = *(char**)stack_low - (char*)start;
    if (index >= tot_size) continue;

#if USE_REF_COUNTS
    Bool_vector *vec;
    if (index < pc_size) {
      vec = pointed_pages;
      index = index/sizeof(Page);
    } else {
      index = (index - pc_size)/sizeof(Core_c);
      vec = pointed_cores;
    }     
#else
    index = index/sizeof(Page);
#endif
    vec->set(index);
  }
}


void Persistent_cache::free_pages(int num_pages) {  
  // Compute the number of pages we need to discard.
  int pages_to_discard = num_pages - (store->num_free() - FE->min_free_pages); 
  
  // Compute pages in store that are referenced by the stack or registers
  // because these cannot be evicted/compacted.
  if (pages_to_discard > 0) {
    FE_STATS(stack_scan_time.start())
    stack_scan();
 
#if USE_REF_COUNTS
    FE->rot->process_zero_rc_log();
#endif

    FE_STATS(stack_scan_time.stop())

#if USE_REF_COUNTS
    FE->rot->adj_rc_during_trans(store->size()*sizeof(Page));
#endif

    // Resize last compacted if needed.
    last_compacted->allowAutoResize();
  };


#if OPTIMAL_PAGE_REPLACEMENT || LRU_PAGE_REPLACEMENT
  trace_replacement(pages_to_discard);
  return;
#endif

  // Set variable used to avoid integer division.
  th_assert(FE->discard_factor < (1 << Discard_factor_bits), 
	    "Discard factor is too big");
  discard_multiplier = (1 << Discard_factor_bits)/FE->discard_factor;

  // Maximum number of pages we can scan to avoid looping forever.
  int max_scan = store->size();
  while (pages_to_discard > 0) {  
    FE_STATS(number_of_scans++)
    FE_STATS(page_scan_time.start())

    // Toggle cache_decay_status after each full cycle through the cache.
    if (scan_index == 0) cache_decay_status = (cache_decay_status+1)%4;

    // We try to scan "scan_factor" times the number of pages we
    // evict. This ensures that we evict at most "1/scan_factor" of
    // the cache between two scans of the same page.
 
    // Separation between scan pointers.
    int separation = store->size()/FE->num_scan_ptrs;
    int span = FE->scan_factor*pages_to_discard;

    for (int sptr_num = 0; sptr_num < FE->num_scan_ptrs; sptr_num++) {
      int cur = sub(scan_index, separation*sptr_num);
      int last = add(cur, span);
      for (;cur != last; cur = add(cur, 1)) {
	register Frame_info *pinfo = page_frame_info+cur;

        // If page cannot be evicted skip it.
	if (!pinfo->is_evictable()) continue; 
    
        // Uninstalled pages have usage 0; do not need to scan them.
	if (pinfo->none_installed()) { last_scanned->insert(cur,0); continue;}
	
        if (sptr_num && !FE->use_full_scan) {
          // For all scan pointers except 0:
	  // Scan for pages with large numbers of uninstalled objects 
          // without decaying usage.
	  unsigned num_objects = page(cur)->num_objects();
	  if ((num_objects-pinfo->num_inst())*FE->discard_factor > num_objects) {
	    // This is a good page to compact (it has a large fraction of 
	    // uninstalled objects). Compute its usage.
	    unsigned usage = 0;
	    survival_fraction(usage, num_objects, pinfo->num_inst()); 
	    last_scanned->insert(cur, usage);  
	  }
	} else {
	  // Compute page usage; decay it; and insert it in the set of last 
          // pages scanned. 
          unsigned usage = collect_usage_and_decay(cur, !sptr_num 
		   && page_decay_status[cur] != cache_decay_status);
	  last_scanned->insert(cur, usage);
	}
      } 
    }

    FE_STATS(page_scan_time.stop())
   
    scan_index = add(scan_index, span);
    max_scan -= span;

    last_scanned->end_scan();
    
    FE_STATS(page_compaction_time.start())
    pages_to_discard -= compact(pages_to_discard, FE->use_eviction);
    FE_STATS(page_compaction_time.stop())

    // We scanned all pages and we still could not evict enough pages. 
    if (pages_to_discard && max_scan <= 0) 
      th_fail("Error: Persistent cache is too small");
  }
}


int Persistent_cache::compact_page(Last_compacted_entry *dest_entry, 
				   Page *source, unsigned threshold) {
  // Breaking into the page abstraction.
  th_assert(dest_entry == 0 || dest_entry->p != source, "Invalid arguments");

  // Fetch data into registers for speed:  
  Resident_object_table *rot = FE->rot;
  OR_num orx = source->get_or();

  // Whether or not to decay usage while compacting. 
  long decay_val = (page_decay_status[index(source)] != cache_decay_status) ? 1 : 0;

  // The next variables are used to iterate over the objects in source.
  Slot *sdata = source->data;
  Uint sosize = source->header.osize;
  Offset *soffsets = source->header.offsets;
  Uint sindex = 0;
  Frame_info *sinfo = page_frame_info + index(source);

  // Page id of last compacted page I encountered for which there is
  // no intact page in the cache. Used to avoid lookups in page map.
  Uint no_intact = sinfo->id();

  // If source is intact, set intact to true and mark source as evicted.
  bool intact = false;  
  if (no_intact != 0) { intact = true; mark_evicted(sinfo, orx); }

  // Page id of the last object evicted. Used to avoid lookups in compacted map.
  Uint last_evicted_pid = 0;

  // Counter for the number of objects retained in an intact page
  // or discarded in a compacted page.
  int obj_count = 0;

  // Initialize pointer to destination page.
  Page *dest;
  if (dest_entry == 0) {
    dest = 0;
    dest_entry = new Last_compacted_entry;
    last_compacted->store(orx, dest_entry);
  } else {
    dest = dest_entry->p;
  }
  int *dest_hist = dest_entry->usage_hist;

    
  while (1) {  
    if (dest == 0) {
      // Set up source to compact objects within it.
      compact_inside_source(source, sindex, sinfo, dest_hist, intact);
      dest = source;
    }

    // Fetch data into registers for speed.
    int dbytes = dest->bytes_available();
    Slot *dslot = dest->data + dest->header.min_offset;
    Slot *ddata = dest->data;
    Offset *doffsets = dest->header.offsets;
    Uint dindex = dest->header.osize;

    for (; sindex < sosize; sindex++) {
      if (!sinfo->is_installed(sindex))  continue;

      // If object is installed, we need to process it.
      Fields f = (Fields) (sdata + soffsets[sindex]);
      Core c = rot->fetch(f->get_handle());
      Uint page_id = Oref_page(c->oref());
      Uint tot_slots = c->num_slots()+Fields_c::header_slots();
	
      // If there is a cached non-compacted copy of this object's
      // page in the PC, move the object to that copy of the
      // page. This can only happen if source is a compacted page.
      if (no_intact != page_id) {
	if (move_to_page(orx, page_id, c)) continue;
        else no_intact = page_id;
      }
  
      // Note that we cannot evict objects that were modified 
      // in the current transaction.
      Uint obj_usage = f->get_usage();
      if ((obj_usage > threshold) | f->is_written()) {
	FE_STATS(tot_objects_compacted++)
	
        // If there is no space in the destination page for this object 
	// exit loop. Note that this can only happen if source != dest.
        dbytes = dbytes - tot_slots*sizeof(Slot) - sizeof(Offset);
	if ((dbytes < 0) | (dindex > Max_onum)) break;

        // Decay usage if needed.
	obj_usage = (obj_usage + decay_val) >> decay_val;
        f->set_usage(obj_usage);

        // Update usage histogram for compacted page.
	dest_hist[obj_usage]++;
	  
        dslot -= tot_slots;
	doffsets[dindex++] = dslot - ddata;
	c->set_fields((Fields) dslot);

	Slot *d = dslot;
	Slot *s = (Slot*)f;
	Slot *max = (Slot*)f+tot_slots;
        if ((dslot < s) | (dslot >= max)) 
          // Use fast copy path when there is no overlapping.
	  while (s < max) *d++ = *s++;
        else
	  // Use memmove to handle overlapping correctly.
	  memmove(dslot, f, tot_slots*sizeof(Slot)); 
      } else {
	// Evict object. 
	tot_objects_evicted++;

	// Mark object as evicted and adjust reference counts. 
	rot->evict_object(c, orx);

        // All installed objects in intact pages are guaranteed not to be cached
        // in compacted pages therefore we continue here.
	if (intact) continue;
        
        // Object is from the same page as last one; increment the count.
	if (last_evicted_pid == page_id) { obj_count++; continue;}

        // This object is from a new page; update the compacted map entry of
        // the old page and set bookeeping for next page.
	cpmap->decrement(orx, last_evicted_pid, obj_count);
	last_evicted_pid = page_id;
	obj_count = 1;
      }
    }
  
    if (intact) obj_count += dindex - dest->header.osize;
    else cpmap->decrement(orx, last_evicted_pid, obj_count);

    // Update dest data.
    dest->header.min_offset = dslot - ddata;
    Frame_info *dinfo = page_frame_info + index(dest);
    if (dest->header.osize == 0) dinfo->clear();
    dinfo->mark_installed_range(dest->header.osize, dindex);
    dest->header.osize = dindex;
       
    // Check if compaction is over.      
    if (sindex == sosize) {
      // TODO: If count == 0 need to inform OR that page was discarded.
      // Maybe we shoukd do this lazily by piggybacking this info in 
      // an invalidation ack?
      if (intact && obj_count > 0) cpmap->increment(orx, no_intact, obj_count);

      // The source page was freed and there is still space in dest.
      if (dest != source) return -1;

      // Dest got full but the source page was freed.
      if (source->header.osize == 0) {
	last_compacted->remove_fast(orx);
	delete dest_entry;
	return -1;
      }

      // Source was not freed and became the last compacted page.
      dest_entry->p = source;
      return source->bytes_available();
    }
    
    // Objects did not fit in dest; Go back and compact them within source. 
    // Insert usage of compacted page in last_scanned.
    unsigned page_usage = 0;
    usage_value(page_usage, dest_hist, dest->header.osize);
    unsigned index_dest = index(dest);
    last_scanned->insert(index_dest, page_usage);

    // Update page_decay_status to indicate we should not decay the usage
    // of the objects in this page until the next cycle. 
    page_decay_status[index_dest] = cache_decay_status;
    dest = 0;
  }
}


bool Persistent_cache::move_to_page(OR_num orx, Uint page_id, Core c) {
  Page *p = pmap->lookup(orx, page_id);
  if (p == 0) return false;

  FE_STATS(tot_objects_merged++)
  Frame_info *pinfo = page_frame_info+index(p);
  th_assert(Oref_page(c->oref()) == page_id, "Invalid page");
  Uint onum = Oref_index(c->oref());
  Uint tot_bytes = (c->num_slots()+Fields_c::header_slots()) * sizeof(Slot);

  // Copy object's fields into p.
  Slot *dslot = p->fast_lookup(onum);
  memcpy(dslot, c->get_fields(), tot_bytes);
  c->set_fields((Fields) dslot);
  
  // Record that the page has a new installed object.
  pinfo->mark_installed(onum);
   
  // Mark object as uncached in a compacted page.
  cpmap->decrement(orx, page_id, 1);

  return true;
}


unsigned Persistent_cache::compact(unsigned pages_to_discard, bool evict) {
  unsigned short index;       // Index of page being compacted.
  unsigned short usage;       // Usage of page being compacted.

  unsigned int discarded = 0; // Number of pages discarded in this scan.

  // Remove the pages with the worst usage from scanned set and compact them.
  // If evict is true, we only retain objects that were modified during
  // the current transaction. Otherwise, we retain objects with usage
  // higher than threshold. The threshold value is obtained from the
  // page usage and ensures that 1/FE->discard_factor of the page
  // is freed (if the page was modified or was accessed after being scanned
  // it is possible that only a smaller fraction of the page will be freed).
  while (last_scanned->remove(index, usage)) {
    Page *source = page(index);
    int free_bytes = -1;

    // If page is no longer evictable then skip it.
    Frame_info *pinfo = page_frame_info+index;  
    if (!pinfo->is_evictable()) continue;

    OR_num orx = source->get_or();
    Last_compacted_entry* dest;

    // If page can be referenced from the stack or registers then
    // it cannot be compacted.
    if (stack_refs->test(index)) continue;

    if (!pinfo->none_installed()) {
      FE_STATS(tot_pages_compacted++)

      // Fetch last page that we compacted from this or.
      if (!last_compacted->find(orx, dest)) {
	  dest = 0;
      } else {
	if (dest->p == source) {
	  last_compacted->remove_fast(orx);
	  delete dest;
	  dest = 0;
	}
      }

      // Compact "good" objects in source. 
      unsigned threshold = (evict) ? Max_usage + 1 :  usage >> Extra_usage_bits;
      free_bytes = compact_page(dest, source, threshold);
    } else {
      FE_STATS(tot_pages_dis_wo_comp++)
      mark_evicted(pinfo, orx);
    }

    if (free_bytes == -1) {
      free_page(source);
      if (++discarded == pages_to_discard) break;
    }
  }
 
  return discarded;
}


void Persistent_cache::mark_evicted(Frame_info *pinfo, OR_num orx) {
  th_assert(pinfo->id() != 0, "Invalid page id");
  FE_STATS(num_intact_pages--)

  Uint page_id = pinfo->id();

  // First, remove page from the pmap because it is no longer cached.
  pmap->remove(orx, page_id);

  // Second, remove the page from or_map
  OR_info *or_info = FE->or_map->lookup(orx);
  Uint seg_id = Segid_from_pageid(page_id);
  Seg_info *seg_info = or_info->lookup_seg_info(seg_id);
  
  seg_info->cached_pages.reset(Pagenum_from_pageid(page_id));

  pinfo->set_id(0);
}




void Persistent_cache::compact_inside_source(Page *p, Uint index, Frame_info *sinfo, 
                                             int *hist, bool intact) {
  int max  = p->header.osize;
  p->header.min_offset = Page_slots;
  p->header.osize = 0;
  
  // Initialize usage histogram for compacted page.
  for (Uint i=0; i < Max_usage+1; i++) hist[i] = 0;

  if (!intact)  return;
  
  // If page is not compacted sort the offsets array in decreasing
  // order of offsets (for a compacted page, the offsets array is already 
  // sorted in the right order). We only need to sort the portion of the 
  // offsets array we did not look at. Sorting is required to ensure that
  // we do not overwrite objects we did not process when compacting.
  Offset *o = p->header.offsets;
  for (int i = index+1; i < max; i++) {
    for (int j = i-1; j >= (int)index; j--) {
      if (o[j] >= o[j+1]) break;
      // swap values.
      Offset tmp = o[j]; o[j] = o[j+1]; o[j+1] = tmp;
      // swap sinfo values.
      sinfo->swap_installed(j, j+1);
    }
  }
}
