#include <string.h>
#include <stream.h>

#include "compiler/C++wrappers/fields.h"
#include "compiler/C++wrappers/core.h"

#include "runtime/stats.h"
#include "runtime/fe_send_message.h"
#include "runtime/or_recv_message.h"
#include "runtime/or_info.h"

#include "or_recv_data_msg.h"
#include "persistent_cache.h"
#include "scanned_set.h"
#include "compacted_map.h"
#include "resident_object_table.h"
#include "surrogate_table.h"
#include "page_map.h"
#include "optimal_replacer.h"

#include "fe/main/fe_config.h"

#include "config/vdefs/OPTIMAL_PAGE_REPLACEMENT.h"
#include "config/vdefs/LRU_PAGE_REPLACEMENT.h"

#include "common/locator.h"
#include "common/objrefs.h"

#include "utils/communication.h"
#include "utils/th_assert.h"
#include "utils/fail.h"
#include "utils/sort.h"
#include "utils/array.h"
#include "utils/mdebug.h"
#include "utils/Monitor.h"

#include "frame_info.h"


OR_obj *Persistent_cache::fetch(OR_num orx, Oref oref) {
  // Lookup page of <or,oref> in the cache.
  Fields f=0;
  OR_recv_data_msg *recv_msg=0;
  Page *p = pmap->lookup(orx, Oref_page(oref));
  if (p == 0 || (f = (Fields) p->lookup(Oref_index(oref))) == 0 || f->is_invalid()) {
    FE_STATS(num_fetches++)
    FE_STATS(fetch_time.start())
    Request_id id = 0;
    Frame_info *pinfo = 0;

    // XXX Added ! in front of pinfo since page is already on its way if 
    // it is not null and not evictable (not evictable is marked on 
    // receiving a response to the fetch msg from the OR)
    if (p && !(pinfo = page_frame_info + index(p))->is_evictable()) {
	// The page is already on its way
	// Note: Invariant: Pages in the pmap are always evictable EXCEPT
	// when that page is in the pipeline from the OR
	// Extract the message id and fix the pinfo id to be the pageid now
	id = pinfo->id();
	pinfo->set_id(Oref_page(oref));
    } else {
	// Object <or,oref> not cached, must fetch its page from the or.
	Uint num_pages_req;
	Seg_bmap pgroup = get_prefetch_group(orx, Oref_page(oref), num_pages_req);

	// Create send and receive message objects.
	// Do we  ever not ask for alloc rights?
	FE_send_fetch_msg send_msg(oref, pgroup, TRUE);
	recv_msg = new OR_recv_data_msg;

	if (!FE->background_replacement) free_pages(num_pages_req);

	FE_STATS(net_fetch_time.start());
	FE->comm->send(OR_address(orx), &send_msg, recv_msg, id);

	if (FE->background_replacement) free_pages(num_pages_req);
    }

    // Block waiting for fetch reply.
    if (FE->comm->handle_messages_until_done(id) < 1) th_fail("Fetch failed\n");
    if (recv_msg && recv_msg->remove()) delete recv_msg;

    // Page should now be cached.
    p = lookup_page(orx, Oref_page(oref));
    th_assert(p, "Fetch failed");

    // Lookup requested object. 
    f = (Fields) p->lookup(Oref_index(oref));
    th_assert(f, "Invalid page fetched");

    // If fetched object was invalidated try to fetch it again.
    if (f->is_invalid()) return fetch(orx, oref);

#if OPTIMAL_PAGE_REPLACEMENT || LRU_PAGE_REPLACEMENT
    trace_rep->add_page(orx, Oref_page(oref));
#endif

    FE_STATS(fetch_time.stop())    
  }

  // Increment no. of installed objects in the page, and
  // if this is the first installation in page, the no. of pages installed
  Frame_info *pinfo = page_frame_info + index(p);
  if (pinfo->none_installed()) {
    num_pages_installed++;
    FE_STATS(tot_num_pages_installed++)
  }
  th_assert(!pinfo->is_installed(Oref_index(oref)), "Object was already installed");

  OR_obj *res = (OR_obj*) f;
  if (!res->is_surrogate()) pinfo->mark_installed(Oref_index(oref));
  return res;
}


Oref Persistent_cache::get_root(OR_num ornum) {
  FE_send_get_root_msg send_msg;
  OR_recv_root_msg recv_msg; 
  FE->comm->send_recv(OR_address(ornum), &send_msg, &recv_msg);
  return recv_msg.or_root;
}


Seg_bmap Persistent_cache::get_prefetch_group(OR_num orx, Uint page_id,
					      Uint &num_req) {
  Seg_bmap fetch_map(FALSE);
  Uint page_num = Pagenum_from_pageid(page_id);

  // If user wants to fetch one page per fetch.
  if (FE->max_prefetch == 1) {
    num_req = 1;
    fetch_map.set(page_num);
    return fetch_map;
  }

  // ... else user wants to fetch more than one page per fetch.
  const int min_pref_group = 1;
  const int max_pref_group = Max_pages_in_seg;
  static bool full_cache = false;

  // Use a larger threshold when the cache fills up to start fetching less.
  if (!full_cache && store->num_free() < Max_pages_in_seg+1) {
    full_cache = true;
    pref_group_sz = 1;
  }

  float threshold = (full_cache) ? FE->high_threshold : FE->low_threshold;
 
  float ratio = ((float) num_pages_installed) / ((float) num_pages_fetched);

  // Change prefetch group size based on ratio. Ratio is a crude
  // measure of the quality of recent prefetches.
  pref_group_sz = (ratio > threshold) ? pref_group_sz + 1 : pref_group_sz - 1;
  
  // We could use a multiplicative rule as the one bellow experiment with it.
  // It seems to be too aggressive.
  // pref_group_sz = (int)(ratio * pref_group_sz * (threshold + 2.0));

  // If user wants fixed size prefetch groups use fixed size.
  if (FE->max_prefetch > 0) pref_group_sz = FE->max_prefetch;

  // Ensure that the prefetch group size is within bounds
  if (pref_group_sz > max_pref_group) 
    pref_group_sz = max_pref_group;
  else if (pref_group_sz < min_pref_group) 
    pref_group_sz = min_pref_group;

  FE_STATS(fetch_group_buckets[pref_group_sz]++)

  // Fetch bitmap with pages from the segment that are already cached.
  OR_info *or_info = FE->or_map->lookup(orx);
  Uint seg_id = Segid_from_pageid(page_id);
  Seg_info *seg_info = or_info->lookup_seg_info(seg_id);

  //
  // Compute fetch bitmap:
  // We set bits for pref_group_sz pages starting in the bit corresponding 
  // to page_id. Unless the group wraps around the end of the segment, 
  // then we set the bits for the last pref_group_sz pages in the segment.
  //
  Uint forward = pref_group_sz + page_num;
  if (forward > Max_pages_in_seg) forward = Max_pages_in_seg;
  num_req = 0;

  for (Uint i = forward - pref_group_sz; i < forward; i++) {
    if (!seg_info->cached_pages.test(i)) {
	// If page is not cached include in prefetch group
        fetch_map.set(i);
        num_req++;
     }
  }

  // Even if requested page is cached we may receive a full page in
  // the reply.
  if (!fetch_map.test(page_num)) num_req++;

  return fetch_map;
}


void Persistent_cache::update_prefetch_estimates(int num_fetched) {
  const int Bad_memory_factor = 2;

  FE_STATS(tot_num_pages_fetched+=num_fetched)

  // Update adaptive prefetching statistics.
  float ratio = ((float) num_pages_installed) / ((float) num_pages_fetched);

  num_pages_fetched = num_pages_fetched/Bad_memory_factor + pref_group_sz;

  num_pages_installed = (int)(num_pages_installed/Bad_memory_factor + 
                               (pref_group_sz - num_fetched) * ratio);
}

void Persistent_cache::merge_pages(OR_num orx, Uint pageid,
				   Page* new_page) {
  
  // Fixed to handle invalidations correctly
  
  // XXX This code assumes that no OR GC has happened
  // Breaking into the page abstraction
  
  Page* orig = lookup_page(orx, pageid);
  th_assert(orig, "Original page must exist for the merge");
  // Move the objects from the new to the old page
  Page::Slow_iter iter(orig);
  Slot *obj;
  Uint onum, num_slots;
  // Copy the unused objects in orig from new_page
  while (iter.get(onum, obj, num_slots)) {
    Fields f = (Fields) obj;
    if (!f) continue;
    
    if (f->is_invalid()) {
      // The existing page has an invalid object
      // so we have to update it in the old page
      Fields new_obj = (Fields) new_page->fast_lookup(onum);
      memcpy(f, new_obj, num_slots * Slot_size);

      // Don't have to install the object, taken care by
      // the caller.   XXX is that for sure??
      // When DISPATCH_D is called, a residency check is
      // performed, which checks whether the object has been marked as 
      // discarded in the ROT.
      // If so it calls Make_resident, which in turn calls pc->fetch.
      // If the page and fields are found, the FE doesn't fetch from the 
      // OR, but only makes the object resident (changes the class id to
      // point to the ROT entry, and changes the ROT entry to point to 
      // the object.)	
      continue;
    }
    
  }
  
  if (new_page->num_objects() == orig->num_objects()) {
    // We're finished with the new_page and don't need it any more
    free_page(new_page);
    return;
  }
  
  // There are new objects on newpage that need to be copied
  
  // Definitely assuming no GC XXX
  // Breaking the page abstraction completely. Aaaarrrggghhh!!
  
  // Copy the objects from the new_page to intact page
  int old_min_offset = orig->header.min_offset;
  int new_min_offset = new_page->header.min_offset;
  th_assert(old_min_offset > new_min_offset, "Bad min offset in new
	page");
  Slot *new_objs = &new_page->data[new_min_offset];
  int copy_slots = old_min_offset - new_min_offset;
  memcpy(&orig->data[new_min_offset], new_objs, copy_slots *
	 Slot_size);
  
  // Copy the offsets array and set the min_offsets, size also
  int old_size = orig->num_objects();
  int new_size = new_page->num_objects();
  th_assert(new_size > old_size, "New size must be greater than
	old_size");
  memcpy(&orig->header.offsets[old_size],
	 &new_page->header.offsets[old_size],
	 (new_size - old_size)*sizeof(Offset));
  orig->header.min_offset = new_min_offset;
  orig->header.osize = new_size;
  
  // We're finished with the new_page and don't need it any more
  free_page(new_page);
}

void Persistent_cache::merge_intact_pages(OR_num orx, Uint pageid,
					  Page* new_page) {
    // XXX This code assumes that no OR GC has happened
    // Breaking into the page abstraction

    Page* orig = lookup_page(orx, pageid);
    th_assert(orig, "Original page must exist for the merge");
    // Move the objects from the new to the old page
    Page::Iter iter(orig);
    Slot *obj;
    int onum = -1;
    // Copy the unused objects in orig from new_page
    while (iter.get(obj)) {
	onum++;
	Fields f = (Fields) obj;
	if (!f || f->is_invalid()) continue;
	Core c = c = FE->rot->safe_fetch(f->get_handle());
	if (c && c->get_fields() == f) continue; // Object is installed

	// Copy the object from the new_page to the existing one
	int num_slots = c->num_slots() + Fields_c::header_slots();
	Fields new_obj = (Fields) new_page->fast_lookup(onum);
	memcpy(f, new_obj, num_slots * Slot_size);
    }
    if (new_page->num_objects() == orig->num_objects()) return;
    // There are new objects on newpage that need to be copied

    // Definitely assuming no GC XXX
    // Breaking the page abstraction completely. Aaaarrrggghhh!!

    // Copy the objects from the new_page to intact page
    int old_min_offset = orig->header.min_offset;
    int new_min_offset = new_page->header.min_offset;
    th_assert(old_min_offset > new_min_offset, "Bad min offset in new page");
    Slot *new_objs = &new_page->data[new_min_offset];
    int copy_slots = old_min_offset - new_min_offset;
    memcpy(&orig->data[new_min_offset], new_objs, copy_slots * Slot_size);

    // Copy the offsets array and set the min_offsets, size also
    int old_size = orig->num_objects();
    int new_size = new_page->num_objects();
    th_assert(new_size > old_size, "New size must be greater than old_size");
    memcpy(&orig->header.offsets[old_size], 
	   &new_page->header.offsets[old_size],
	   (new_size - old_size)*sizeof(Offset));
    orig->header.min_offset = new_min_offset;
    orig->header.osize = new_size;
}


void Persistent_cache::register_pages(OR_num orx, Uint seg, Seg_bmap sent, 
	      Seg_bmap alloc_rights, Seg_bmap empty, Page **fetched_pages) {
  OR_info *or_info = FE->or_map->lookup(orx);
  Seg_info * si = or_info->lookup_seg_info(seg);

  // Register allocation rights received for the segment.
  si->alloc_rights = si->alloc_rights | alloc_rights;
  si->empty = empty;

  // Iterate over fetched pages inserting them in the pmap and handling any
  // necessary merges.
  Uint page_num;
  int i = 0;
  Seg_bmap::Iter iter(&sent, TRUE);

  while (iter.get(page_num)) {
    Page *p = fetched_pages[i++];
    Uint page_id = Pageid_from_segid(seg, page_num); 
    Uint pindex = index(p);

    // Mark page as uninstalled and set its or correctly.
    p->set_or(orx);
    if (si->alloc_rights.test(page_num))
	or_info->add_alloc_rights(page_id, p);

    // Init page frame information to contain the page's id,
    // mark page as evictable, and mark all objects in page as uninstalled.
    page_frame_info[pindex].set_id(page_id);
    page_frame_info[pindex].mark_evictable();
    page_frame_info[pindex].clear();

    // Set decay status on current page.
    page_decay_status[pindex] = (cache_decay_status-1)%4;

    if (si->cached_pages.test(page_num)) {
      // if another intact version of the page is cached call merge procedure.
      // XXX
      //merge_intact_pages(orx, page_id, p);
      merge_pages(orx, page_id, p);
    } else {
      // Put the page in the map and set its bit in the segment's cache_pages
      FE_STATS(num_intact_pages++)
      pmap->store(page_id, pindex);
      si->cached_pages.set(page_num);
    }
  }    
      
  // Update prefetch estimates.
  FE->pc->update_prefetch_estimates(i);
}


