#ifndef _FRAME_INFO_H
#define _FRAME_INFO_H

#include "utils/bits.h"
#include "utils/basic.h"

//
// This file must be only used by the Peristent cache code
//

// Instantiate bitmap class used to record which objects in a page frame are
// installed.
#define BITMAP_CLASS Installed
#define BITMAP_SIZE  Max_onum+1
#include "utils/bitmap.h"

const Uint Obj_all_refs_set = 0xffffffff;

static const Uint Inst_bit_off = 1;
static const Uint Id_bit_off = Onum_bits + Inst_bit_off;
static const Uint Max_id_bits = 32 - Id_bit_off;
static const Uint Max_id = (1 << Max_id_bits) - 1;

class Frame_info {
  // Overview: Stores information about a page frame.
public:
  Frame_info() { id_inst_evictable = 0; }
  // Effects: create page frame describing empty frame.

  bool is_evictable() const { return id_inst_evictable & 0x01; }
  // Effects: Returns true iff the page in this frame is evictable.

  void mark_evictable() { id_inst_evictable |= 0x01; }
  // Effects: Mark the page in this frame as evictable.

  void mark_unevictable() { id_inst_evictable &= ~0x01; }
  // Effects: Mark the page in this frame as unevictable.

  void set_id(Uint id) { 
    th_assert(id <= Max_id, "Invalid argument");
    id_inst_evictable = (id << Id_bit_off) | 
      (id_inst_evictable & ((1 << Id_bit_off)-1));
  }
  // Effects: Mark this frame as holding the page with identifier "id".  

  Uint id() const { return  id_inst_evictable >> Id_bit_off; }
  // Effects: Returns the identifier of the page in this frame.

  Uint num_inst() const { return (id_inst_evictable >> Inst_bit_off) & Max_onum; }
  // Effects: Return number of objects installed in this page frame.

  void set_num_inst(Uint num) {
    th_assert(num <= Max_onum, "Invalid argument");
    id_inst_evictable = (id_inst_evictable & ~(Max_onum << Inst_bit_off))
      | (num << Inst_bit_off);
  }
  // Effects: Set the number of installed objects in this page.

  bool is_installed(Uint index) const { return installed.test(index);}
  // Effects: Check if the object with index "index" in this page frame
  //   is installed.

  void mark_installed(Uint index) { 
    installed.set(index); 
    set_num_inst(num_inst()+1);
  }
  // Effects: Mark the object with index "index" in this page frame as installed.

  void mark_uninstalled(Uint index) { 
    installed.reset(index);
    set_num_inst(num_inst()-1);
  }
  // Effects: Mark  the object with index "index" in this page frame as uninstalled.

  void swap_installed(Uint i, Uint j) {
    bool tmp = installed.test(i);
    installed.assign(i, installed.test(j));
    installed.assign(j, tmp);
  }
  // Effects: Swap the installed status of the objects with indices i and j.
   
  void mark_installed_range(Uint min, Uint max) { 
    installed.set_range(min, max);
    set_num_inst(num_inst()+max-min);
  }
  // Effects: Mark objects with indices in [min, max[ as installed.

  bool none_installed() {return installed.is_all_false(); }
  // Effects: Returns true iff no object in this page frame is installed. 

  void clear() { installed.clear(); set_num_inst(0); }
  // Effects: Mark all objects in this page frame as uninstalled. 


private:
  // The following is a bitfield with the following fields:
  // Page id:32-Onum_bits-1 | inst:Onum_bits | evictable:1
  // id is the id of the page on the corresponding frame, or 0
  // if the page was compacted. inst is the number of installed objects 
  // in the page. Evictable is non-null iff page the page
  // canot be evicted because it has already been freed, it is being used or
  // it was borrowed.
  Uint id_inst_evictable;  

  // Bitmap with a bit set for each object in the page frame that 
  // is installed.
  Installed installed;
};

#endif /* _FRAME_INFO_H */
