#include <stdio.h>
#include <stream.h>
#include "common/xref_set.h"
#include "fe/main/fe_config.h"
#include "storage_arena.h"
#include "persistent_cache.h"
#include "lru_replacer.h"
#include "utils/stack_refs.h"

LRU_replacer::LRU_replacer(Stack_refs const *s) : sr(s) { 
  cached_list = 0;
 
  // Consume first entry (0,0) from trace.
  Trace_entry t;
  fread(&t, sizeof(t), 1, FE->trace_file);
}

LRU_replacer::~LRU_replacer() { 
  while (cached_list) {
    Trace_list_entry *l = cached_list;
    cached_list = l->next;
    delete l;
  }
}


void LRU_replacer::bring_up_to_date(Uint access_counter) {
  // Read entries in trace file until an entry with value greater
  // than access_counter is found.
  Trace_entry t;
  while (fread(&t, sizeof(t), 1, FE->trace_file) != 0) {
    if (t.last_access > access_counter) {
      fseek(FE->trace_file, -sizeof(t), SEEK_CUR);
      break;
    }

    // Search for page's entry in cached list...
    Trace_list_entry *e1 = cached_list;
    Trace_list_entry *e2 = 0;
    bool found = false;
    while (e1) {
      if (e1->t.page_id == t.page_id) {
	found = true;
	break;
      }
      e2 = e1;
      e1 = e1->next;
    }
    if (!found) th_fail("Page entry was not in cached list");

    // ... remove it.
    if (e2) 
      e2->next = e1->next;
    else
      cached_list = e1->next;

    // Update page entry to contain new access and put it at the head
    // MRU position of cached list.
    e1->t.last_access = t.last_access;
    e1->next = cached_list;
    cached_list = e1;
  }
}




Page *LRU_replacer::replace_page() {
  th_assert(cached_list, "Cached list is empty in replace_page");

  // Search for cached page that will be accessed farthest into the
  // future and is not referenced by the stack.
  Trace_list_entry *e1 = cached_list, *e2 = 0;
  int num_elem = 0; 
  while (e1->next) { 
    num_elem++;
    e2 = e1;
    e1 = e1->next;
  }


  Page *p = FE->pc->lookup_page(1, e1->t.page_id);
  th_assert(p != 0, "Non-cached page in cached list");  

  // If last page was referenced from the stack then
  // try previous one.
  while (1) {
    if (!sr->is_referenced(FE->pc->index(p))) {
      e2->next = e1->next;
      delete e1;
      return p;
    }

    num_elem--;
    e1 = cached_list;
    e2 = 0;
    for (int i=0; i < num_elem; i++) {
      e2 = e1;
      e1 = e1->next;
    }
    p = FE->pc->lookup_page(1, e1->t.page_id);
    th_assert(p != 0, "Non-cached page in cached list");  
  }

  return p;
} 



void LRU_replacer::add_page(OR_num orx, Uint page_id) {
  th_assert(orx == 1, "LRU replacer only works for OR 1\n");
  Trace_list_entry *e = new Trace_list_entry;
  e->t.page_id = page_id;
  e->t.last_access = (Uint)-1;
  e->next = cached_list;
  cached_list = e;
}



