#ifndef _Old_ref_log
#define _Old_ref_log 1

#include "compiler/C++wrappers/obj.h"

#include "runtime/trans_log.h"

#include "common/shortp.h"

#include "fe/main/fe_config.h"

#include "utils/bool_vector.h"

class Old_ref_log { 
  //
  // Overview: Keeps a log of the old references in copies of objects
  //   that were modified during the current transaction.
  //
public:
  Old_ref_log(Trans_log *tlog, bool persistent);
  // Effects: Creates an empty log for references in modified persistent objects
  //   (if persistent is true) or modified volatile objects (otherwise) in tlog.

  ~Old_ref_log();
  // Effects: Destroys log.


  void log_reference(Obj_x *pref);
  // Requires: pref is a pointer to a reference ivar in a copy of a 
  //   modified persistent or volatile object object (depending on the 
  //   the value of the persistent flag when the log was created).
  //   Can only be used if no objects have been added (or removed) to tlog 
  //   after this was created or grow was called.
  // effects: Marks the reference pointed to by pref as stale. 


  void grow();
  // effects: grows log to account for modified objects that were added
  //    to tlog since this was created.

  class Old_ref_iter {
    //  An iterator for yielding the stale swizzled references in
    //  original copies of persistent objects modified during the
    //  current transaction.
    //
    //  Once created, an iterator must be used and destroyed before any
    //  objects are added to the write set of tlog.
    //  The effect is undefined if an iterator method is called after
    //  such a change.
	
  public:
    Old_ref_iter(Old_ref_log* olog);
    // effects Return an iterator over the set of stale refs in olog

    ~Old_ref_iter();

    bool get(Core& c);
    // modifies "c"
    // effects  Sets "c" to the next stale reference in a copy of an object
    //   modified by the current transaction and returns true. Returns false 
    //   if there are no more objects (and does not modify c)

  private:
    Old_ref_log* olog; // Transaction log for which elements are yielded;
    Bool_vector::Iter *iter;
  };

private:
  friend Old_ref_iter;


  Data *copies;             // A pointer to the array of copies in translog.
  Bool_vector *mod_refs;    // Bitmap with one bit for each Slot in 
                            // copies; a bit is set iff the 
                            // the corresponding Slot contains a swizzled
                            // reference that was modified during the current
                            // transaction.
};



inline Old_ref_log::Old_ref_log(Trans_log *tlog, bool persistent) {
  copies = (persistent) ? &tlog->persistent_copies : &tlog->volatile_copies;
  int size = (persistent)  ? FE->per_undo_log_size : copies->size()*2;
  mod_refs = new Bool_vector(size, false);
}

inline Old_ref_log::~Old_ref_log() { delete mod_refs; }

inline void Old_ref_log::log_reference(Obj_x *pref) {
  th_assert((int)mod_refs->size() >= copies->size(), "New objects were added to log");
  mod_refs->set((Slot*)pref-copies->as_pointer());
}

inline void Old_ref_log::grow() {
  if ((int)mod_refs->size() < copies->size()) {
    Bool_vector *tmp =  new Bool_vector(copies->size()*2, false);
    Bool_vector::Iter iter(mod_refs, true);
    unsigned int index;
    while (iter.get(index)) tmp->set(index);
    delete mod_refs;
    mod_refs = tmp;
  }
}

inline Old_ref_log::Old_ref_iter::Old_ref_iter(Old_ref_log *o_log) {
    olog = o_log;
    iter = new Bool_vector::Iter(olog->mod_refs, true); 
}


inline Old_ref_log::Old_ref_iter::~Old_ref_iter() { delete iter; }

inline bool Old_ref_log::Old_ref_iter::get(Core& c) {
    Uint index;
    if (!iter->get(index)) return false;
  
    Shortp sp = (Shortp) (olog->copies->slot(index));
    c =  (Core)SHORTP_AS_FULL_V(sp);
    return true;
}


#endif // _Old_ref_log



