#include <stdio.h>
#include "common/xref_set.h"
#include "fe/main/fe_config.h"
#include "storage_arena.h"
#include "persistent_cache.h"
#include "optimal_replacer.h"
#include "utils/stack_refs.h"

implementArray(AccessesArray, Uint)
implementOpenHashMap(Trace_cache, Uint, Trace_cache_entry*, hash_int, Uint_equal)

declareArray(Trace_entryArray, Trace_entry)
implementArray(Trace_entryArray, Trace_entry)

#define QUICK_SORT_ELEMENT Trace_entry
#include "utils/impl_sort.t"

Optimal_replacer::Optimal_replacer(Stack_refs const *s) : sr(s) {
  cached_list = 0;
  trace_cache1 = new Trace_cache;
  last_access_cached1 = 0;
  trace_cache2 = new Trace_cache;
  last_access_cached2 = 0;

  read_next_chunk(trace_cache1, last_access_cached1);
  read_next_chunk(trace_cache2, last_access_cached2);
}

Optimal_replacer::~Optimal_replacer() { 
  while (cached_list) {
    Trace_list_entry *l = cached_list;
    cached_list = l->next;
    delete l;
  }

 Trace_cache::Bindings gen1(trace_cache1);
  for (; gen1.ok(); gen1.next()) {
    Trace_cache_entry *c = gen1.val();
    delete c;
    gen1.del();
  }
  delete trace_cache1;

  Trace_cache::Bindings gen2(trace_cache2);
  for (; gen2.ok(); gen2.next()) {
    Trace_cache_entry *c = gen2.val();
    delete c;
    gen2.del();
  }
  delete trace_cache2;
}

void Optimal_replacer::read_next_chunk(Trace_cache *dest, Uint &last_access) {
  // First, clear any previous content of dest.
  Trace_cache::Bindings gen(dest);
  for (; gen.ok(); gen.next()) {
    Trace_cache_entry *c = gen.val();
    delete c;
    gen.del();
  }
  
  // Read next chunk of trace file into dest.
  Trace_entry t;
  int num_entries = 0;
  while (num_entries < Trace_window) {
    Trace_cache_entry *c;
    
    if (fread(&t, sizeof(t), 1, FE->trace_file) == 0) {
      last_access = (Uint) -1;
      return;
    }
    
    num_entries++;
    if (dest->fetch(t.page_id, c)) {
      c->accesses.append(t.last_access);
    } else {
      c = new Trace_cache_entry;
      c->accesses.append(t.last_access);
      dest->store(t.page_id, c);
    }
  }

  last_access = t.last_access;
}



Trace_entry Optimal_replacer::search_trace_caches(Uint access_counter, 
						  Trace_entry &stale) {
  Trace_entry t;
  Trace_cache_entry *c;
  bool found = false;  
  Trace_cache *tc = trace_cache1;

  for (int i=0; i < 2; i++) {
    if (tc->fetch(stale.page_id, c)) {
      int max = c->accesses.size();
      for (; c->accesses_cursor < max; c->accesses_cursor++) {
	if (c->accesses[c->accesses_cursor] > access_counter) {
	  found = true;
	  break;
	}
      }
    }
    if (found) break;
    tc = trace_cache2;
  }

  if (found) 
    t.last_access = c->accesses[c->accesses_cursor];
  else 
    t.last_access = (Uint) -1;

  t.page_id = stale.page_id;

  return t;
}


void Optimal_replacer::bring_up_to_date(Uint access_counter) {
  // Update trace caches if needed.
  if (access_counter > last_access_cached1) {
    Trace_cache *tmp = trace_cache1;
    trace_cache1 = trace_cache2;
    last_access_cached1 = last_access_cached2;
    trace_cache2 = tmp;
    read_next_chunk(trace_cache2, last_access_cached2);
  }

  // Check which entries in the beggining of cached_list
  // are stale (i.e. entries such that last_access is smaller 
  // than access_counter) and use trace caches to update them.
  Trace_entryArray new_entries;
  
  while (cached_list != 0) {
    if (cached_list->t.last_access <= access_counter) {
      // If entry i is stale: search for next trace entry 
      // for the same page in trace caches and append it to new_entries.
      new_entries.append(search_trace_caches(access_counter, cached_list->t));

      // Remove entry from cached list.
      Trace_list_entry *e = cached_list;
      cached_list = cached_list->next;
      delete e;
    } else
	break;
  }

  // Sort new_entries by increasing value of last_access.
  quick_sort(new_entries.as_pointer(), new_entries.size());

  // Merge new_entries into cached_list.
  int new_entries_max = new_entries.size();
  Trace_list_entry *e1 = cached_list, *e2 = 0;
  for (int i=0; i < new_entries_max; i++) {
    Trace_list_entry *e = new Trace_list_entry;
    e->t = new_entries[i];

    while (e1 && e1->t.last_access < e->t.last_access) {
      e2 = e1; 
      e1 = e1->next; 
    }

    if (e2) {
      e->next = e1;
      e2->next = e;
    } else {
      e->next = cached_list;
      cached_list = e;
    }

    if (!e1) e1 = e; else e2 = e;
  }

}



Page *Optimal_replacer::replace_page() {
  th_assert(cached_list, "Cached list is empty in replace_page");

  // Search for cached page that will be accessed farthest into the
  // future and is not referenced by the stack.
  Trace_list_entry *e1 = cached_list, *e2 = 0;
  int num_elem = 0; 
  while (e1->next) { 
    num_elem++;
    e2 = e1;
    e1 = e1->next;
  }

  Page *p = FE->pc->lookup_page(1, e1->t.page_id);
  th_assert(p != 0, "Non-cached page in cached list");  

  // If last page was referenced from the stack then
  // try previous one.
  while (1) {
    if (!sr->is_referenced(FE->pc->index(p))) {
      e2->next = e1->next;
      delete e1;
      return p;
    }

    num_elem--;
    e1 = cached_list;
    e2 = 0;
    for (int i=0; i < num_elem; i++) {
      e2 = e1;
      e1 = e1->next;
    }
    p = FE->pc->lookup_page(1, e1->t.page_id);
    th_assert(p != 0, "Non-cached page in cached list");  
  }

  return p;
} 



void Optimal_replacer::add_page(OR_num orx, Uint page_id) {
  th_assert(orx == 1, "Optimal replacer only works for OR 1\n");
  Trace_list_entry *e = new Trace_list_entry;
  e->t.page_id = page_id;
  e->t.last_access = 0;
  e->next = cached_list;
  cached_list = e;
}



