
#include <string.h>
#include <stream.h>

#include "compiler/C++wrappers/fields.h"
#include "compiler/C++wrappers/core.h"

#include "runtime/stats.h"
#include "runtime/fe_send_message.h"
#include "runtime/or_recv_message.h"
#include "runtime/or_info.h"

#include "or_recv_data_msg.h"
#include "persistent_cache.h"
#include "scanned_set.h"
#include "compacted_map.h"
#include "resident_object_table.h"
#include "surrogate_table.h"
#include "page_map.h"
#include "optimal_replacer.h"
#include "lru_replacer.h"
#include "frame_info.h"

#include "fe/main/fe_config.h"

#include "config/vdefs/OPTIMAL_PAGE_REPLACEMENT.h"
#include "config/vdefs/LRU_PAGE_REPLACEMENT.h"

#include "common/locator.h"
#include "common/objrefs.h"

#include "utils/communication.h"
#include "utils/th_assert.h"
#include "utils/fail.h"
#include "utils/sort.h"
#include "utils/array.h"
#include "utils/mdebug.h"
#include "utils/Monitor.h"

// Define maps.

#define IMPL_KEY   UIntKey
#define IMPL_VALUE Last_compacted_entry *
#include "utils/impl_map.t"

// The type of Last compacted is defined in or_info.cc so there is no
// need to create another instance of this map.


Persistent_cache::Persistent_cache(Storage_arena *sa) : store(sa) {
  Uint num_pages = store->size();

  pmap  = new Page_map(num_pages, this); // Average six pages per bucket.     
  cpmap = new Compacted_map(num_pages); // Average three pages per bucket.

  scan_index = 0;
  page_frame_info = new Frame_info[num_pages];

  stack_refs = new Bool_vector(num_pages, false);

  last_scanned = new Scanned_set(FE->scans_remembered); 
  last_compacted = new Last_compacted(1); // Start with single or.   

  num_pages_fetched = 1;   
  num_pages_installed = 1;                    
  pref_group_sz = 1;

  page_decay_status = new char[num_pages];
  for (Uint i=0; i < num_pages; i++) page_decay_status[i] = 0;
  cache_decay_status = 0;

  reset_statistics();
#if OPTIMAL_PAGE_REPLACEMENT 
  trace_rep = new Optimal_replacer(store);
#endif

#if LRU_PAGE_REPLACEMENT 
  trace_rep = new LRU_replacer(store);
#endif
}


Persistent_cache::~Persistent_cache(void) {
    delete pmap;		// Pages don't need to be deleted.
    delete cpmap;
    delete [] page_frame_info;
    delete last_scanned;
    MapGenerator<UIntKey, Last_compacted_entry *> glcu(*last_compacted);
    UIntKey key;
    Last_compacted_entry *lcu;
    while (glcu.get(key, lcu)) delete lcu;
    delete last_compacted;
    delete stack_refs;
    delete store;
    delete page_decay_status;
#if OPTIMAL_PAGE_REPLACEMENT || LRU_PAGE_REPLACEMENT
    delete trace_rep;
#endif
}


void Persistent_cache::print_stats(ostream &o) {
  o << "\nPersistent cache stats: \n"
    << " Net fetch time (time between sending fetch and receiving reply): "
    << net_fetch_time.elapsed()
    << "\n Total fetch time (including pc bookkeeping): "
    << fetch_time.elapsed()
    << "\n Stack scan time: " << stack_scan_time.elapsed()
    << "\n Page scan time: " << page_scan_time.elapsed()
    << "\n Page compaction time: " << page_compaction_time.elapsed()
    << "\n Number of scans (equals number of compactions): " << number_of_scans
    << "\n Number of pages scanned: " << num_pages_scanned
    << "\n Number of pages compacted: " << tot_pages_compacted
    << "\n Number of pages discarded w/o compaction: " << tot_pages_dis_wo_comp
    << "\n Number of objects compacted: " << tot_objects_compacted
    << "\n Number of objects evicted: " << tot_objects_evicted
    << "\n Number of objects merged: " << tot_objects_merged << endl;

  cpmap->print_stats(o);

  o << "\n Histogram of usage of scanned objects: \n";
  Uint i;
  for (i = 0; i <= Max_usage; i++) o << " " << usage_buckets[i];
  o << "\n";
  o << "\n Number of non-installed: " << num_not_installed << "\n";

  o << "\n Number of fetches: " << num_fetches
    << "\n Number of pages fetched: " << tot_num_pages_fetched
    << "\n Number of pages installed: " << tot_num_pages_installed
    << "\n Number of background fetches: " << num_delayed_fetches
    << "\n Pages in foreground fetches: " << tot_fore_pages
    << "\n Pages in background fetches: " << tot_back_pages
    << "\n Fetches for compacted pages: " << fetches_for_compacted << endl;

  o << "\n Histogram of prefetch group sizes: \n";
  for (i=0; i <= Max_pages_in_seg; i++) o << " " << fetch_group_buckets[i]; 
  o << "\n";
  
  o << "\n Total number of pages: " << store->size()
    << "\n Number of borrowed pages: " << num_borrowed_pages 
    << "\n Number of intact pages: " << num_intact_pages << "\n\n";
}

void Persistent_cache::reset_statistics() {
  static bool first_time = true;
  net_fetch_time.reset();
  fetch_time.reset();

  stack_scan_time.reset();
  page_scan_time.reset();
  page_compaction_time.reset();

  number_of_scans = 0;
  num_pages_scanned = 0;
  tot_pages_compacted = 0;
  tot_objects_compacted = 0;
  tot_objects_merged = 0;
  tot_objects_evicted = 0;

  num_fetches = 0;
  num_delayed_fetches = 0;
  tot_fore_pages = 0;
  tot_back_pages = 0;
  tot_num_pages_fetched = 0;
  tot_num_pages_installed = 0;
  fetches_for_compacted = 0;

  Uint i; 
  for (i=0; i <= Max_usage; i++)
    usage_buckets[i] = 0;

  for (i=0; i <= Max_pages_in_seg; i++)
    fetch_group_buckets[i] = 0; 

  if (first_time) {
    num_not_installed = 0;
    num_borrowed_pages = 0;
    num_intact_pages = 0;
    first_time = false;
  }
}


Fields Persistent_cache::lookup(OR_num orx, Oref oref) const {
  Page *p = pmap->lookup(orx, Oref_page(oref));
  if (p != 0) {
    Fields f = (Fields) p->lookup(Oref_index(oref));
    if (f != 0 && !f->is_invalid()) return f;
  }
  return 0;
}


Page *Persistent_cache::lookup_page(OR_num orx, Uint page_id) const {
  return pmap->lookup(orx, page_id);
}


void Persistent_cache::invalidate(Core c) {
  // If object is in compacted page decrement number of cached objects for
  // the page in the compacted map.
  Fields f = c->get_fields();
  OR_num orx = f->orx();
  Uint page_id = Oref_page(c->oref());
  cpmap->decrement(orx, page_id, 1);
 
  // Mark object as uninstalled in its page frame.
  Page *p = f->page();
  page_frame_info[index(p)].mark_uninstalled(Oref_index(c->oref()));

  // Mark object fields as invalid.
  f->mark_invalid();

  // Mark any other copy of the fields in an intact page as invalid.
  f = lookup(orx, c->oref());
  if (f) f->mark_invalid();
}


Page *Persistent_cache::alloc_and_lock_page(OR_num orx, Uint page_id) {
  Page *p = borrow_page();
  Uint pindex = index(p);
  p->set_or(orx);
  pmap->store(page_id, pindex);
  page_frame_info[pindex].set_id(page_id);
  page_frame_info[pindex].clear();
  return p;
}


void Persistent_cache::lock_page(Page *p) {
  page_frame_info[index(p)].mark_unevictable();
}


void Persistent_cache::unlock_page(OR_num orx, Uint page_id) {
  Page *p = lookup_page(orx, page_id);
  th_assert(p, "Invalid argument");
  page_frame_info[index(p)].mark_evictable();
}

Page * Persistent_cache::alloc_page() {
  while (true) {
    Page *p = store->alloc_page();

    if (p != 0) {
      page_frame_info[index(p)].mark_evictable();
      return p;
    }
    
    // Try to free pages by discarding objects
    free_pages(Max_pages_in_seg);
  }
}

void Persistent_cache::free_page(Page *p) {
  // Ensure the invariant that all free pages are unevictable and have
  // a null id. OR_info and free_pages depend on this invariant.
  page_frame_info[index(p)].mark_unevictable();
  page_frame_info[index(p)].set_id(0);
  store->free_page(p);
}



Page * Persistent_cache::borrow_page(void) {
  FE_STATS(num_borrowed_pages++)
  Page *p = alloc_page();
  th_assert(p != 0, "Unable to allocate page.");
  page_frame_info[index(p)].mark_unevictable();
  return p;
}


void Persistent_cache::return_page(Page *p) { 
  FE_STATS(num_borrowed_pages--)
  store->free_page(p);
}


Fields Persistent_cache::allocate(OR_num orx, Oref oref) {
    Page * p = lookup_page(orx, Oref_page(oref));
    th_assert(p != 0, "Allocating in non-cached page");
    th_assert(page_frame_info[index(p)].id() == Oref_page(oref), 
	      "Allocating in compacted page");

    Fields f = (Fields) p->lookup(Oref_index(oref));
    th_assert((f != NULL), "PC unable to lookup field of newly persistent object.");

    // Increment num installed objects in page.
    page_frame_info[index(p)].mark_installed(Oref_index(oref));
    
    return (f);
}

Oref Persistent_cache::get_new_oref(OR_num orx, int num_slots) {
    OR_info *or_info = FE->or_map->lookup(orx);
    Oref oref = or_info->get_new_oref(num_slots);
    if (oref != NULL_OREF) return oref;

    // Must send alloc message
    FE_send_alloc_msg send_msg;
    OR_recv_alloc_reply_msg recv_msg;
    FE->comm->send_recv(OR_address(orx), &send_msg, &recv_msg);
    oref = or_info->get_new_oref(num_slots);
    th_assert(oref != NULL_OREF, "Allocation failed even after alloc message");
    return oref;
}
